//
//  PSMLanSession.h
//  BombSquad
//
//  Created by John Pannell on 2/7/10.
//  Copyright 2010 Positive Spin Media. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import "PSMLanDefinitions.h"
#import "PSMLanProtocols.h"
@class PSMLanListener;

/*
 
 The PSMLanSession mirrors the interface and functionality of Game Kit's GKSession, but uses LAN connections instead of bluetooth to transport messages between peers.
 
 */

@interface PSMLanSession : NSObject <PSMLanListenerDelegate, PSMLanPeerDelegate, NSNetServiceDelegate, NSNetServiceBrowserDelegate> {
    
    // delegate
    id                  delegate;               // helps me get stuff done
    id                  dataHandler;            // where to send the data that I get from peers
    void                *dataContext;           // context that came with the data handler
    
    // internal
    NSMutableSet        *availablePeers;        // list of all peers who showed up and are visible via Bonjour
    NSMutableDictionary *peerLookup;            // collection of objects representing other sessions; peerID is key
    NSMutableSet        *pendingPeers;          // peers holding tank - ones that have yet to resolve
    
    // identity
    NSString            *sessionID;             // unique string to identify Bonjour service
    NSString            *_peerID;               // unique string to identify myself as a peer
    NSString            *displayName;           // human readable name for myself as a peer
    PSMLanSessionMode   sessionMode;            // server, client, or peer
    
    // state
    BOOL                available;              // can someone else connect to me?
    NSTimeInterval      disconnectTimeout;      // how long I'll wait before disconnecting unresponsive peers
    
    // Bonjour - advertising my presence
    NSString            *serviceType;           // Bonjour service name; based on sessionID
    NSNetService        *broadcaster;           // exposed Bonjour net service advertising my availability
    NSNetServiceBrowser *seeker;                // looking for other peers
    PSMLanListener      *listener;              // listening for connection requests

}

@property(assign) id<PSMLanSessionDelegate> delegate;
@property(getter=isAvailable) BOOL available;
@property(assign) NSTimeInterval disconnectTimeout;
@property(readonly) NSString *displayName;
@property(readonly) NSString *peerID;
@property(readonly) NSString *sessionID;
@property(readonly) NSMutableSet *availablePeers;
@property(readwrite) PSMLanSessionMode sessionMode;

// creation/destruction
- (id)initWithSessionID:(NSString *)newID displayName:(NSString *)name sessionMode:(PSMLanSessionMode)mode;

// directions for connecting to peers
// (sending request)
- (void)connectToPeer:(NSString *)peerID withTimeout:(NSTimeInterval)timeout;
- (void)cancelConnectToPeer:(NSString *)peerID;
// (responding to a request)
- (BOOL)acceptConnectionFromPeer:(NSString *)peerID error:(NSError **)error;
- (void)denyConnectionFromPeer:(NSString *)peerID;
// (done with a connection)
- (void)disconnectFromAllPeers;
- (void)disconnectPeerFromAllPeers:(NSString *)peerID;

// information about other peers
- (NSString *)displayNameForPeer:(NSString *)peerID;
- (NSArray *)peersWithConnectionState:(PSMLanPeerConnectionState)state;

// data transmission
- (BOOL)sendData:(NSData *)data toPeers:(NSArray *)peers withDataMode:(PSMLanSendDataMode)mode error:(NSError **)error;
- (BOOL)sendDataToAllPeers:(NSData *)data withDataMode:(PSMLanSendDataMode)mode error:(NSError **)error;
- (void)setDataReceiveHandler:(id)handler withContext:(void *)context;

@end
