//**********************************************************************************************************************
//
//  BXLog.h
//
//  Author:		Peter Baumgartner, peter@baumgartner.com
//  Copyright:	©2005-2015 by Boinx Software, Ltd. & IMAGINE GbR, All rights reserved.
//	Abstract:	Logging convenience functions
//
//**********************************************************************************************************************


#pragma mark HEADERS
    
#import <Foundation/Foundation.h>


//----------------------------------------------------------------------------------------------------------------------


#pragma mark CONSTANTS
    
// BOXLogLevel determines how much information is logged...

typedef NS_ENUM(NSInteger,BXLogLevel)
{
    kLogLevelNone,		// No logging at all
    kLogLevelError,		// Only serious (non-recoverable) errors are logged
    kLogLevelWarning,	// Serious and minor (recoverable) errors are logged
    kLogLevelDebug,		// Errors and degugging information is logged
    kLogLevelVerbose	// Everything is logged - can cause severe performance penalty
};

#ifndef DEFAULT_LOG_LEVEL
#ifdef CONFIGURATION_Debug
#define DEFAULT_LOG_LEVEL kLogLevelWarning
#else
#define DEFAULT_LOG_LEVEL kLogLevelError
#endif
#endif


// The domains group logging into different topics. The log level can be chosen separately for each domain...

#define kDomainCount 128
#define kLogDomainFramework kDomainCount-1

typedef NSInteger BXLogDomain;
extern BXLogLevel sDomainLevel[kDomainCount];


// The destination determines where the log is sent...

enum
{
    kBXLogDestinationNone = 0x0,
    kBXLogDestinationConsole = 0x1,
    kBXLogDestinationFile = 0x2,			// Note: Logging to a file requires BXLogOpen() and BXLogClose() to be called.
    kBXLogDestinationCrashlytics = 0x4, 	// Note: Logging to Crashlytics needs to be activated by calling BXLogCrashlyticsDidInitialize()
};


//----------------------------------------------------------------------------------------------------------------------


#pragma mark FUNCTIONS

// Choose logging destination...

void BXLogSetDestination(NSInteger inDestination);
NSInteger BXLogDestination(void);
void BXLogSetDefaultDestination(void);

// activate Crashlytics logging

void BXLogCrashlyticsDidInitialize(void);

// Set or get domain level...

void BXLogSetDomainLevel(BXLogDomain inDomain,BXLogLevel inLevel);
BXLogLevel BXLogDomainLevel(BXLogDomain inDomain);

// Open/close private logging file. Returns YES if it opened a log file. Returns NO, if it can't
// open a log file or a log file is already open...

BOOL BXLogOpen(NSString* inPath);
void BXLogClose(void);

NSString* BXLogPath(void);				// Get path of private log file
NSString* BXLogContent(void);			// Get contents of private log file

// Designated logging methods...

void BXLogSessionHeader(NSString* inFlavor);
void BXLogError(OSStatus error,const char* message,const char* file,NSInteger line);


//----------------------------------------------------------------------------------------------------------------------


// Convenience macros...

#ifndef BXLog
#define BXLog(format,args...) NSLog(format,##args)
#endif

#ifndef BXLogInDomain
#define BXLogInDomain(domain,level,format,args...) if (sDomainLevel[domain] >= level || level==kLogLevelError) NSLog(format,##args)
#endif

#ifndef BXLogIf
#define BXLogIf(condition,format,args...) if (condition) NSLog(format,##args)
#endif

#ifndef BXLogErr
#define BXLogErr(error) BXLogError(error, NULL, __FILE__, __LINE__)
#endif

#ifndef BXLogErrWithMessage
#define BXLogErrWithMessage(error, message) BXLogError(error, message, __FILE__, __LINE__)
#endif

#ifndef BXLogErrorCode
#define BXLogErrorCode(error) BXLogError(error, NULL, __PRETTY_FUNCTION__, __LINE__)
#endif

#ifndef BXLogErrorMessage
#define BXLogErrorMessage(error,message) BXLogError(error, message, __PRETTY_FUNCTION__, __LINE__)
#endif


//----------------------------------------------------------------------------------------------------------------------
