#include "CustomEnchManager.h"
#include "skse/GameRTTI.h"

#include <Windows.h>

static const char savePath[] = "data/skse/plugins/hdtHighHeelEnch.dat";
static const char PathPlugins[] = "skyrim/plugins.txt";
static const char PathLoadOrder[] = "skyrim/loadorder.txt";
static const char MCMModName[] = "hdtHighHeelMCM.esp";
static const char* PackageName = "hdtHighHeelMCM";

#include "common.h"
#include "log.h"

void CustomEnchManager::NODE::Release()const
{
	TESObjectARMO* armor = DYNAMIC_CAST(LookupFormByID(form), TESForm, TESObjectARMO);
	if(!armor) return;
	armor->enchantable.enchantment = nullptr;
}

template <class T> T CustomEnchManager::read()
{
	T ret;
	if(fread(&ret, 1, sizeof(T), file) != sizeof(T))
		throw -1;
	return ret;
}

template <> std::string CustomEnchManager::read()
{
	size_t len = read<WORD>();
	char buf[1000];
	if(fread(buf, 1, len, file) != len)
		throw -1;
	buf[len] = 0;
	return buf;
}

template <class T> void CustomEnchManager::write(const T& a)
{
	fwrite(&a, 1, sizeof(T), file);
}

template <> void CustomEnchManager::write(const std::string& a)
{
	write<WORD>(a.size());
	fwrite(a.c_str(), 1, a.length(), file);
}

void CustomEnchManager::clear()
{
	for(auto& i : enchInfo)
		for(auto& j : i.forms)
			j.Release();

	enchInfo.clear();
	modlist.clear();
}

void CustomEnchManager::save()
{
	file = fopen(savePath, "wb");

	write<BYTE>(modlist.size());
	for(auto i = modlist.begin(); i<modlist.end(); ++i)
		write(*i);

	write(enchInfo.size());
	for(int i=0; i<enchInfo.size(); ++i)
	{
		auto ench = DYNAMIC_CAST( LookupFormByID(enchInfo[i].enchFormID), TESForm, EnchantmentItem );
		if(!ench || !ench->effectItemList.count || !ench->effectItemList[0])
		{
			LogWarning("Error Enchantment %08x ... skip", enchInfo[i].enchFormID);
			continue;
		}
		write(ench->effectItemList[0]->magnitude);
		write(enchInfo[i].forms.size());
		for(auto& form : enchInfo[i].forms)
			write(form);
	}

	fclose(file);
}

#include <shlobj.h>
#include <iostream>
#include <fstream>
void ReadCurrentModList(std::vector<std::string>& modlist)
{
	modlist.clear();
	modlist.reserve(256);

	char buf[1000];
	SHGetFolderPath(0, CSIDL_LOCAL_APPDATA, 0, 0, buf);
	std::string appdata = buf;
	if(appdata.back() != '/' || appdata.back() != '\\')
		appdata.push_back('\\');
	std::string pluginsPath = appdata+PathPlugins;
	std::string orderPath = appdata+PathPlugins;

	std::unordered_set<std::string> plugins;

	std::ifstream fin(orderPath);
	while(!fin.eof())
	{
		std::string tp;
		std::getline(fin, tp);
		if(tp.empty()) continue;
		modlist.push_back(tp);
	}
	fin.close();

	fin.open(pluginsPath);
	while(!fin.eof())
	{
		std::string tp;
		std::getline(fin, tp);
		if(tp.empty()) continue;
		plugins.insert(tp);
	}
	fin.close();

	for(auto i = modlist.begin(); i<modlist.end();)
		if(plugins.find(*i) == plugins.end())
			modlist.erase(i);
		else ++i;
}

void CustomEnchManager::load()
{
	file = fopen(savePath, "rb");
	if(!file) return;

	try{

		std::vector<std::string> oldList;
	
		int modnum = read<BYTE>();
		LogInfo("Reading modlist from save(%d)", modnum);
		for(int i=0; i<modnum; ++i)
			oldList.push_back(read<std::string>());
		
		LogInfo("Reading modlist from game");
		ReadCurrentModList(modlist);
		int map[256];
		for(auto i = oldList.begin(); i<oldList.end(); ++i)
		{
			auto idx = modlist.begin();
			for(; idx < modlist.end(); ++idx)
				if(!stricmp(idx->c_str(), i->c_str()))
					break;
		
			int oldidx = i-oldList.begin();
			int newidx = idx-modlist.begin();
			if(idx < modlist.end())
			{
				LogRaw("\tMapping mod (%02x) -> (%02x) : %s", oldidx, newidx, i->c_str() );
				map[oldidx] = newidx;
			}
			else
			{
				LogRaw("\tDeleted mod (%02x) : %s", oldidx, i->c_str() );
				map[oldidx] = -1;
			}
		}

		int numType = read<int>();
		if(numType < enchInfo.size())
			LogWarning("More enchantments(%d) than saved(%d)", enchInfo.size(), numType);
		else if(numType > enchInfo.size())
		{
			LogWarning("Less enchantments(%d) than saved(%d)", enchInfo.size(), numType);
			numType = enchInfo.size();
		}

		for(int i=0; i<numType; ++i)
		{
			enchInfo[i].forms.clear();
			auto ench = DYNAMIC_CAST( LookupFormByID(enchInfo[i].enchFormID), TESForm, EnchantmentItem);
			read(ench->effectItemList[0]->magnitude);
			int count = read<int>();
			for(int j=0; j<count; ++j)
			{
				int form = read<int>();
				int mod = (unsigned)form >> 24;
			
				if(map[mod] != -1)
				{
					form = form & 0x00FFFFFF | (map[mod]<<24);
					auto armor = DYNAMIC_CAST(LookupFormByID(form), TESForm, TESObjectARMO);
					enchant(armor, i);
				}
			}
		}
	} catch(...) {
		LogWarning("save broken!! clear all list");
		for(auto& i : enchInfo)
		{
			for(auto& j : i.forms)
				j.Release();
			i.forms.clear();
		}
	}

	fclose(file);
}

void CustomEnchManager::enchant(TESObjectARMO* form, int type)
{
	if(!form) return;

	if(type >= enchInfo.size() || type<0)
	{
		LogError("Failed to enchant %dth enchantment : Out Of Range", type);
		return;
	}

	remove(form);

	auto ench = DYNAMIC_CAST( LookupFormByID(enchInfo[type].enchFormID), TESForm, EnchantmentItem );
	if(!ench)
	{
		LogWarning("Failed to enchant %08x", form->formID);
		return;
	}

	enchInfo[type].forms.insert(form->formID);
	form->enchantable.enchantment = ench;
	LogInfo("Enchanting %08x by %08x", form->formID, enchInfo[type].enchFormID);
}

void CustomEnchManager::remove(TESObjectARMO* form)
{
	if(!form) return;

	form->enchantable.enchantment = nullptr;
	for( auto& i : enchInfo )
		i.forms.erase(form->formID);
}

void CustomEnchManager::addEnchType(EnchantmentItem* item)
{
	if(!item) return;
	if(!item->effectItemList.count || !item->effectItemList[0])
	{
		LogWarning("Error enchantment %08x ... skip", item->formID);
		return;
	}

	enchInfo.push_back(CustomEnch());
	enchInfo.back().enchFormID = item->formID;
	LogInfo("Custom enchantment manager : New enchantment added - %08x", item->formID);
}

static CustomEnchManager EnchManager;

void EnchManagerClear(StaticFunctionTag*)
{
	EnchManager.clear();
	LogInfo("Custom enchantment manager : clear");
}

void EnchManagerLoad(StaticFunctionTag*)
{
	LogInfo("Custom enchantment manager : loading");
	EnchManager.load();
}

void EnchManagerSave(StaticFunctionTag*)
{
	LogInfo("Custom enchantment manager : saving");
	EnchManager.save();
}

void RegisterCustomEnch(StaticFunctionTag*, EnchantmentItem* ench)
{
	EnchManager.addEnchType(ench);
}

void SetEnchNthMagnitude(StaticFunctionTag*, EnchantmentItem* ench, UInt32 idx, float val)
{
	if(!ench || idx<0 || idx>ench->effectItemList.count || !ench->effectItemList[idx]) return;
	ench->effectItemList[idx]->magnitude = val;
}

void AttachEnchantment(StaticFunctionTag*, TESObjectARMO* armor, UInt32 idx)
{
	EnchManager.enchant(armor, idx);
}

void RemoveEnchantment(StaticFunctionTag*, TESObjectARMO* armor)
{
	EnchManager.remove(armor);
}

void CustomEnchManager::RegisterFunctions(VMClassRegistry* registry)
{
	AutoRegisterFunction(EnchManagerClear, PackageName, registry);
	AutoRegisterFunction(EnchManagerLoad, PackageName, registry);
	AutoRegisterFunction(EnchManagerSave, PackageName, registry);
	AutoRegisterFunction(RegisterCustomEnch, PackageName, registry);
	AutoRegisterFunction(SetEnchNthMagnitude, PackageName, registry);
	
	AutoRegisterFunction(AttachEnchantment, PackageName, registry);
	AutoRegisterFunction(RemoveEnchantment, PackageName, registry);
}