#include "Adapter.h"

static const char* PackageName = "hdtHighHeelEffect";

static const BSFixedString NINODE_NPC = "NPC";
static const BSFixedString NINODE_LFoot = "NPC L Foot [Lft ]";
static const BSFixedString NINODE_RFoot = "NPC R Foot [Rft ]";
static const BSFixedString NINODE_LCalf = "NPC L Calf [LClf]";
static const BSFixedString NINODE_RCalf = "NPC R Calf [RClf]"; 
static const BSFixedString NINODE_LRot = "hdtHighHeel Lft";
static const BSFixedString NINODE_RRot = "hdtHighHeel Rft";

#include <cstdint>
#include <vector>

#include "configs.h"
#include "Thread.h"
#include "geometry.h"
#include "../skse/GameRTTI.h"

NiAVObject* GetNode(TESObjectREFR* obj, BSFixedString name) throw(...)
{
	if(!obj || !obj->loadedState || !obj->loadedState->node) throw -1;
	NiAVObject* ret = obj->loadedState->node->GetObjectByName(&name.data);
	if(!ret) throw -1;
	return ret;
}

Actor* GetRef(int formID) throw(...)
{
	auto form = DYNAMIC_CAST( LookupFormByID(formID), TESForm, Actor);
	if(!form) throw -1;
	return DYNAMIC_CAST( form, TESForm, Actor);
}

float ClampRate(Actor* actor, float rate, float lastRate)
{
	UInt64 flag = actor->actorState.flags04 | (UInt64)(actor->actorState.flags08)<<32;
	if(flag & ActorState::kState_Swimming)
		return 0;

	float delta = Configs::stableModeMaxSwing * 0.01f;
	switch( Configs::stableMode )
	{
	case 2:
		STABLE_MODE:
		rate = min(max(rate, lastRate-delta), lastRate+delta);
		break;

	case 1:
		if( flag & ActorState::kState_Moving)
			goto STABLE_MODE;

	default:
		break;
	};
	return rate;
}

struct Node
{
	Node(int id, float m):formID(id), magnitude(m), lastRate(1), lastHeight(0){}

	void Update() throw(...)
	{
		auto akActor = GetRef(formID);

		NiAVObject* lft = GetNode(akActor, NINODE_LFoot);
		NiAVObject* rft = GetNode(akActor, NINODE_RFoot);
		NiAVObject* lcalf = GetNode(akActor, NINODE_LCalf);
		NiAVObject* rcalf = GetNode(akActor, NINODE_RCalf);
		NiAVObject* npc = GetNode(akActor, NINODE_NPC);
		
		float lleglen = (Vector(lcalf->m_worldTransform.pos) - lft->m_worldTransform.pos).Magnitude();
		float rleglen = (Vector(rcalf->m_worldTransform.pos) - rft->m_worldTransform.pos).Magnitude();
		float cosl = (lcalf->m_worldTransform.pos.z - lft->m_worldTransform.pos.z) / lleglen;
		float cosr = (rcalf->m_worldTransform.pos.z - rft->m_worldTransform.pos.z) / rleglen;
		float rate = ClampRate( akActor,  max( max(cosl, cosr) , 0 ), lastRate );
		lastRate = rate;

		static const float CONST_CALF_L = 55.89939793f;
		float calfScale = (lleglen+rleglen) / CONST_CALF_L;
		calfScale *= npc->m_localTransform.scale;
		calfScale /= npc->m_worldTransform.scale; //World -> Object
		float actualheightDelta = calfScale * magnitude;

		float currentHeightDelta = rate * actualheightDelta; //+ Configs::offset;
		npc->m_localTransform.pos.z += currentHeightDelta - lastHeight;
		lastHeight = currentHeightDelta;
		NiAVObject::ControllerUpdateContext ctx;
		npc->UpdateWorldData(&ctx);
	}

	void Recover()
	{
		try{
			
			auto actor = GetRef(formID);
			auto npc = GetNode(actor, NINODE_NPC);
			npc->m_localTransform.pos.z -= lastHeight;
			NiAVObject::ControllerUpdateContext ctx;
			npc->UpdateWorldData(&ctx);
			
			LogDebug("object released ... 0x%08x", formID);

		} catch(...){}
	}

	int formID;
	float magnitude;
	float lastRate;
	float lastHeight;
};

volatile bool RuningFlg;
std::vector<Node> objList;
Thread t;
Lockable lock;
Event NonEmpty;
Event Freeze;
int menuOpenCount = 0;

void CheckEmpty()
{
	if(objList.empty())
		NonEmpty.reset();
	else NonEmpty.set();
}

float GetMagnitude(ActiveEffect* object)
{
	return object->magnitude;
}

void KeepUpdate(StaticFunctionTag*, TESObjectREFR* akActor, float magnitude)
{
	if(!akActor) return;
	
	int id = akActor->formID;
	
	Locker locker(lock);
	auto idx = objList.begin();
	for(; idx < objList.end(); ++idx)
	{
		if(idx->formID == id)
			break;
	}

	if(idx >= objList.end())
	{
		objList.push_back(Node(id, magnitude));
		LogDebug("object added ... 0x%08x", id);
	}
	else idx->magnitude = magnitude;

	CheckEmpty();
}

void StopUpdate(StaticFunctionTag*, TESObjectREFR* akActor)
{
	if(!akActor) return;
	
	Locker locker(lock);
	auto idx = objList.begin();
	for(; idx < objList.end(); ++idx)
	{
		if(idx->formID == akActor->formID)
			break;
	}
	if(idx >= objList.end())
		return;

	idx->Recover();
	objList.erase(idx);
	CheckEmpty();
}

void AdapterClear(StaticFunctionTag*)
{
	Locker locker(lock);
	for(auto& i : objList)
		i.Recover();
	objList.clear();
	NonEmpty.reset();
}

void OnUpdate()
{
	static const HANDLE handles[] = {Freeze.GetHandle(), NonEmpty.GetHandle()};
	WaitForMultipleObjects(2, handles, true, INFINITE);
	
	Locker locker(lock);
	for(auto i = objList.begin(); i<objList.end();)
	{
		try
		{
			i->Update();
			++i;
		}
		catch(...)
		{
			LogWarning("release 0x%08x, resource unloaded", i->formID);
			objList.erase(i);
		}
	}
	
	CheckEmpty();
}

class FreezeEventHandler : public BSTEventSink <MenuOpenCloseEvent>
{
public:
	virtual EventResult		ReceiveEvent(MenuOpenCloseEvent * evn, EventDispatcher<MenuOpenCloseEvent> * dispatcher);
};

EventResult FreezeEventHandler::ReceiveEvent(MenuOpenCloseEvent * evn, EventDispatcher<MenuOpenCloseEvent> * dispatcher)
{
	Locker locker(lock);

	if( evn->opening )
	{
		if(menuOpenCount++ == 1)
		{
			LogDebug("menu open, suspending...");
			Freeze.reset();
		}
	}
	else
	{
		if(--menuOpenCount == 1)
		{
			LogDebug("menu close, resuming...");
			Freeze.set();
		}
	}
	
	return kEvent_Continue;
}

FreezeEventHandler g_freezeEventHandler;

#include "../skse/GameMenus.h"
void RegisterEventSinks(void)
{
	MenuManager * mm = MenuManager::GetSingleton();
	if (mm)
		mm->MenuOpenCloseEventDispatcher()->AddEventSink(&g_freezeEventHandler);
}

bool Adapter::RegisterFunctions(VMClassRegistry * registry)
{
	AutoRegisterFunction(GetMagnitude, PackageName, registry);
	AutoRegisterFunction(KeepUpdate, PackageName, registry);
	AutoRegisterFunction(StopUpdate, PackageName, registry);
	AutoRegisterFunction(AdapterClear, PackageName, registry);
	
	RegisterEventSinks();

	return true;
}

#include <string>
int Adapter::Initialize()
{
	objList.reserve(1000);
	RuningFlg = true;
	CheckEmpty();

	t.func = [](void*){
		
		LogInfo("thread begin...");
		
		while(RuningFlg)
		{
			OnUpdate();
			Sleep(Configs::decalUpdate);
		}
		
		LogInfo("thread end...");
	};
	return t.run(Configs::priority);
}

void Adapter::Release()
{
	RuningFlg = false;
	Freeze.set();
	NonEmpty.set();
	LogInfo("waiting... %d", t.get_handle());
	t.wait();
}