/*
 * Decompiled with CFR 0.152.
 */
package de.nx42.maps4cim.map.relief.srtm;

import com.google.common.collect.Table;
import com.google.common.io.Resources;
import de.nx42.maps4cim.ResourceLoader;
import de.nx42.maps4cim.map.Cache;
import de.nx42.maps4cim.util.gis.Area;
import de.nx42.maps4cim.util.xml.Network;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.UnknownHostException;
import java.text.ParseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TileDownload {
    private static final Logger log = LoggerFactory.getLogger(TileDownload.class);
    protected final Table<Integer, Integer, DownloadURL> downloadMapping;
    protected final Cache cache = new Cache();

    public TileDownload() {
        this.downloadMapping = TileDownload.loadMapping();
    }

    protected static Table<Integer, Integer, DownloadURL> loadMapping() {
        InputStream serialized = ResourceLoader.getMappingSRTM();
        try {
            Table mapping = (Table)ResourceLoader.deserializeObject(serialized);
            serialized.close();
            return mapping;
        }
        catch (FileNotFoundException e) {
            log.error("The serialized srtm-mapping was not found in the classpath!", e);
        }
        catch (ClassNotFoundException e) {
            log.error("The srtm-mapping could not be casted to the guava Table datastructure.", e);
        }
        catch (IOException e) {
            log.error("Error while accessing the serialized srtm-mapping.", e);
        }
        throw new RuntimeException("Could not load SRTM Download-URL mapping!");
    }

    public boolean exists(double lat, double lon) {
        return this.downloadMapping.contains(this.floor(lat), this.floor(lon));
    }

    public boolean exists(int lat, int lon) {
        return this.downloadMapping.contains(lat, lon);
    }

    public File[][] getTiles(Area ar) throws SocketTimeoutException, IOException, UnknownHostException {
        int minLat = this.floor(ar.getMinLat());
        int maxLat = (int)Math.ceil(ar.getMaxLat());
        int minLon = this.floor(ar.getMinLon());
        int maxLon = (int)Math.ceil(ar.getMaxLon());
        int sizeLat = maxLat - minLat;
        int sizeLon = maxLon - minLon;
        if (minLon > maxLon) {
            log.warn("Your selection overlaps the east/west border at E180, W180 -> this is currently not supported!");
        }
        File[][] tiles = new File[sizeLat][sizeLon];
        int i = 0;
        while (i < sizeLat) {
            int j = 0;
            while (j < sizeLon) {
                File tile;
                tiles[i][j] = tile = this.getTile(minLat + i, minLon + j);
                ++j;
            }
            ++i;
        }
        return tiles;
    }

    public File getTile(double lat, double lon) throws SocketTimeoutException, IOException {
        return this.getTile(this.floor(lat), this.floor(lon));
    }

    protected int floor(double a) {
        return (int)StrictMath.floor(a);
    }

    public File getTile(int lat, int lon) throws SocketTimeoutException, IOException, UnknownHostException {
        if (!this.exists(lat, lon)) {
            return null;
        }
        String entry = DownloadURL.getFileName(lat, lon);
        if (this.cache.has(entry)) {
            log.debug("SRTM Tile ({},{}) has been loaded from cache.", (Object)lat, (Object)lon);
            return this.cache.get(entry);
        }
        log.debug("Downloading SRTM Tile for ({},{}). It will be stored in cache for later use.", (Object)lat, (Object)lon);
        URL src = this.getDownloadURL(lat, lon);
        File dest = this.cache.allocate(entry);
        Network.downloadToFile(src, dest, 5.0, 2.0);
        return dest;
    }

    protected URL getDownloadURL(int lat, int lon) {
        DownloadURL dl = this.downloadMapping.get(lat, lon);
        String url = dl.getUrl(lat, lon);
        try {
            return new URL(url);
        }
        catch (MalformedURLException e) {
            log.error(String.format("Could not create a valid SRTM download URL for (%s,%s). Result was %s", lat, lon, url), e);
            throw new RuntimeException("Creating of SRTM tile download URL failed");
        }
    }

    protected static SimpleCoord parseCoordinate(String hgtFileName) throws ParseException {
        String parse = hgtFileName.trim();
        int lat = Integer.MIN_VALUE;
        int lon = Integer.MIN_VALUE;
        int i = 0;
        while (i < parse.length()) {
            char c = parse.charAt(i);
            if (c == 'N') {
                lat = TileDownload.getNumAfterIndex(parse, ++i);
            } else if (c == 'S') {
                lat = -TileDownload.getNumAfterIndex(parse, ++i);
            } else if (c == 'E') {
                lon = TileDownload.getNumAfterIndex(parse, ++i);
            } else if (c == 'W') {
                lon = -TileDownload.getNumAfterIndex(parse, ++i);
            }
            ++i;
        }
        if (lat < -90 || lat > 90) {
            throw new ParseException(String.format("Latitude must be between [-90;+90], but is %s", lat), -1);
        }
        if (lon < -180 || lon > 180) {
            throw new ParseException(String.format("Longitude must be between [-180;+180], but is %s", lon), -1);
        }
        return new SimpleCoord(lat, lon);
    }

    protected static int getNumAfterIndex(String s, int start) throws ParseException {
        if (!Character.isDigit(s.charAt(start))) {
            throw new ParseException(String.format("No digit recognized at index %s in %s, just '%s'", start, s, Character.valueOf(s.charAt(start))), start);
        }
        int last = start + 1;
        while (last < s.length()) {
            if (!Character.isDigit(s.charAt(last))) break;
            ++last;
        }
        return Integer.parseInt(s.substring(start, last));
    }

    protected static enum DownloadURL {
        Africa("Africa"),
        Australia("Australia"),
        Eurasia("Eurasia"),
        Islands("Islands"),
        North_America("North_America"),
        South_America("South_America");

        protected static final String base = "http://dds.cr.usgs.gov/srtm/version2_1/SRTM3/";
        protected static final String ext = ".hgt.zip";
        protected final String folder;

        private DownloadURL(String folder) {
            this.folder = folder;
        }

        public String getFolder() {
            return this.folder;
        }

        public String getUrl(int lat, int lon) {
            StringBuilder sb = new StringBuilder(80);
            sb.append(base);
            sb.append(this.folder);
            sb.append('/');
            sb.append(DownloadURL.getNonationNSEW(lat, lon));
            sb.append(ext);
            return sb.toString();
        }

        public String getIndexURL() {
            return base + this.getFolder();
        }

        public File getIndexLocal() {
            try {
                return new File(Resources.getResource("srtm/" + this.folder + ".html").toURI());
            }
            catch (URISyntaxException e) {
                log.error("Could not resolve path to local file", e);
                return null;
            }
        }

        public static String getNonationNSEW(int lat, int lon) {
            StringBuilder sb = new StringBuilder(16);
            sb.append(lat >= 0 ? (char)'N' : 'S');
            sb.append(String.format("%02d", Math.abs(lat)));
            sb.append(lon >= 0 ? (char)'E' : 'W');
            sb.append(String.format("%03d", Math.abs(lon)));
            return sb.toString();
        }

        public static String getFileName(int lat, int lon) {
            return String.valueOf(DownloadURL.getNonationNSEW(lat, lon)) + ext;
        }
    }

    protected static class SimpleCoord {
        public final int lat;
        public final int lon;

        public SimpleCoord(int lat, int lon) {
            this.lat = lat;
            this.lon = lon;
        }
    }
}

