/*
 * Decompiled with CFR 0.152.
 */
package de.nx42.maps4cim;

import com.beust.jcommander.IParameterValidator;
import com.beust.jcommander.IStringConverter;
import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.google.common.base.Splitter;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import de.nx42.maps4cim.MapGenerator;
import de.nx42.maps4cim.ResourceLoader;
import de.nx42.maps4cim.Result;
import de.nx42.maps4cim.config.Config;
import de.nx42.maps4cim.config.ReliefDef;
import de.nx42.maps4cim.config.TextureDef;
import de.nx42.maps4cim.config.bounds.CenterDef;
import de.nx42.maps4cim.config.texture.ColorDef;
import de.nx42.maps4cim.config.texture.EntityDef;
import de.nx42.maps4cim.map.ReliefMap;
import de.nx42.maps4cim.map.TextureMap;
import de.nx42.maps4cim.map.ex.MapGeneratorException;
import de.nx42.maps4cim.map.relief.MandelbrotRelief;
import de.nx42.maps4cim.map.relief.PlanarRelief;
import de.nx42.maps4cim.map.relief.SRTM;
import de.nx42.maps4cim.map.texture.OsmTexture;
import de.nx42.maps4cim.map.texture.SingleTexture;
import de.nx42.maps4cim.util.gis.Coordinate;
import de.nx42.maps4cim.util.xml.Serializer;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.log4j.Appender;
import org.apache.log4j.RollingFileAppender;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Launcher {
    private static final Logger log = LoggerFactory.getLogger(Launcher.class);
    @Parameter(names={"-h", "-?", "--help"}, description="Prints these usage instructions.")
    private boolean help;
    @Parameter(names={"-c", "--config"}, converter=FileConverter.class, validateWith=FileValidator.class, description="Path to the config.xml file that specifies all program options. If the config.xml is used (which is highly recommended), it will override all other options specified through command line args (except for the output path).")
    protected File config;
    @Parameter(names={"-o", "--output"}, converter=FileConverter.class, validateWith=FileValidator.class, description="Path to the file where the resulting map shall be written into. Overwrites any existing file without warning!")
    protected File output;
    @Parameter(names={"-ce", "--center"}, converter=CoordinateConverter.class, validateWith=CoordinateValidator.class, description="The center of the map. This parameter requires exactly two values: latitude and longitude, each as decimal numbers split by a single comma WITHOUT whitespace in between. A valid input would be -c 48.5,11.5")
    protected Coordinate center;
    @Parameter(names={"-e", "--extent"}, converter=DoubleConverter.class, validateWith=DoubleValidator.class, description="The extent of the map in kilometers. As the resulting map will be quadratic, this setting equals the edge length of the map.")
    protected Double extent;
    @Parameter(names={"-hs", "--height-scale"}, converter=DoubleConverter.class, validateWith=DoubleValidator.class, description="The height-scale to apply on the elevation-data of the current map. The default value of 1.0 is usually sufficient, but for large maps a lower scale might be necessary.")
    protected Double heightScale;

    public static void main(String[] args) {
        Launcher.initLogger();
        try {
            Launcher l = new Launcher();
            l.parse(args);
            if (l.config != null) {
                l.runWithConfig();
            } else {
                l.runWithArgs();
            }
        }
        catch (ParameterException e) {
            log.error(e.getMessage());
        }
    }

    public static void initLogger() {
        File logFile = new File(ResourceLoader.getAppDir(), "maps4cim.log");
        Appender appender = org.apache.log4j.Logger.getRootLogger().getAppender("FA");
        RollingFileAppender fa = (RollingFileAppender)appender;
        fa.setFile(logFile.getAbsolutePath());
        fa.activateOptions();
        log.debug("---------- NEW SESSION ----------");
        log.debug("Writing log to {}", (Object)logFile.getAbsoluteFile());
    }

    protected void runWithConfig() {
        try {
            Config conf = Serializer.deserialize(Config.class, this.config);
            File out = this.output == null ? Launcher.getDefaultOutput() : this.output;
            Launcher.runMapGenerator(conf, this.output);
        }
        catch (Exception e) {
            log.error("reading xml config failed", e);
        }
    }

    protected void runWithArgs() {
        if (this.center == null) {
            throw new ParameterException("No valid config.xml or center defined. Can't generate a map based on no information at all...");
        }
        Config c = Launcher.getDefaultConfig();
        c.bounds = new CenterDef(){
            {
                this.centerLat = Launcher.this.center.getLatitude();
                this.centerLon = Launcher.this.center.getLongitude();
                this.extent = this.extent == null ? 8.0 : Launcher.this.extent;
                this.unit = CenterDef.Unit.KM;
            }
        };
        if (this.heightScale != null) {
            c.relief = new ReliefDef(){
                {
                    this.heightScale = Launcher.this.heightScale;
                }
            };
        }
        File out = this.output == null ? Launcher.getDefaultOutput() : this.output;
        Launcher.runMapGenerator(c, out);
    }

    protected void parse(String[] args) throws ParameterException {
        JCommander jc = new JCommander(this);
        jc.parse(args);
        if (this.help || args.length < 1) {
            jc.usage();
        }
    }

    public static void runMapGenerator(Config conf) {
        Launcher.runMapGenerator(conf, new File("maps4cim-generated.map"));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean runMapGenerator(Config conf, File dest) {
        try {
            Stopwatch stopwatch = new Stopwatch();
            stopwatch.start();
            log.info("Map Generator has been started.");
            ReliefMap rf = Launcher.getReliefMap(conf);
            TextureMap tm = Launcher.getTextureMap(conf);
            MapGenerator mg = new MapGenerator(conf, rf, tm);
            try {
                log.debug("Ressources initialized. The map will now be written...");
                Result res = mg.generateMap(dest);
                stopwatch.stop();
                if (res.isSuccess()) {
                    log.info("The map has been successfully generated. Total Time: {}", (Object)stopwatch.toString());
                    log.info("If you plan to publish this map, please attribute correctly. You can copy the following text to do so:\nThis map was created using maps4cim, with data from the OpenStreetMap (\u00a9 OpenStreetMap contributors).");
                    return true;
                }
                log.warn(res.getReport());
                log.warn("Something went wrong, so your map has been generated in fallback mode (probably it's just empty).\nPlease review the errors and post this log in the forums if you don't know how to fix them.");
                return false;
            }
            catch (MapGeneratorException e) {
                return false;
            }
        }
        catch (Exception e2) {
            log.error("Unexpected Exception", e2);
            return false;
        }
    }

    protected static TextureMap getTextureMap(Config conf) {
        TextureDef.TextureSource ts = conf.texture.getSource();
        switch (ts) {
            case osm: {
                return new OsmTexture(conf);
            }
            case none: {
                return new SingleTexture();
            }
        }
        return new SingleTexture();
    }

    protected static ReliefMap getReliefMap(Config conf) {
        ReliefDef.ReliefSource rs = conf.relief.getSource();
        switch (rs) {
            case srtm: {
                return new SRTM(conf);
            }
            case mandelbrot: {
                return new MandelbrotRelief();
            }
            case none: {
                return new PlanarRelief();
            }
        }
        return new PlanarRelief();
    }

    protected static Config getDefaultConfig() {
        return new Config(){
            {
                this.bounds = new CenterDef(){
                    {
                        this.centerLat = 48.401;
                        this.centerLon = 11.738;
                        this.extent = 8.0;
                        this.unit = CenterDef.Unit.KM;
                    }
                };
                this.relief = new ReliefDef(){
                    {
                        this.heightScale = 1.5;
                    }
                };
                this.texture = new TextureDef(){
                    {
                        this.colors = ColorDef.getDefaults();
                        this.entities = EntityDef.getDefaults();
                    }
                };
            }
        };
    }

    protected static File getDefaultOutput() {
        return new File("maps4cim-generated.map");
    }

    public static class CoordinateConverter
    implements IStringConverter<Coordinate> {
        @Override
        public Coordinate convert(String value) {
            Iterator<String> args = Splitter.on(',').trimResults().split(value).iterator();
            double lat = Double.parseDouble(args.next());
            double lon = Double.parseDouble(args.next());
            return new Coordinate(lat, lon);
        }
    }

    public static class CoordinateValidator
    implements IParameterValidator {
        @Override
        public void validate(String name, String value) throws ParameterException {
            block5: {
                try {
                    Iterator<String> argsIt = Splitter.on(',').trimResults().omitEmptyStrings().split(value).iterator();
                    ArrayList<String> args = Lists.newArrayList(argsIt);
                    if (args.size() >= 2) {
                        try {
                            double lat = Double.parseDouble((String)args.get(0));
                            double lon = Double.parseDouble((String)args.get(1));
                            new Coordinate(lat, lon);
                            break block5;
                        }
                        catch (NumberFormatException e) {
                            throw new ParameterException(String.format("One of the values could not be interpreted as decimal: %s", name, value, e.getMessage()));
                        }
                    }
                    throw new ParameterException("You have to specify the center with exactly two numers separated by a single comma, with no whitespace in between, e.g. 48.5,11.5");
                }
                catch (Exception e) {
                    throw new ParameterException(String.format("Parameter %s \"%s\" could not be interpreted as coordinate: %s", name, value, e.getMessage()));
                }
            }
        }
    }

    public static class DoubleConverter
    implements IStringConverter<Double> {
        @Override
        public Double convert(String value) {
            return Double.parseDouble(value);
        }
    }

    public static class DoubleValidator
    implements IParameterValidator {
        @Override
        public void validate(String name, String value) throws ParameterException {
            try {
                Double.parseDouble(value);
            }
            catch (NumberFormatException e) {
                throw new ParameterException(String.format("Parameter %s \"%s\" could not be interpreted as decimal: %s", name, value, e.getMessage()));
            }
        }
    }

    public static class FileConverter
    implements IStringConverter<File> {
        @Override
        public File convert(String value) {
            return new File(value);
        }
    }

    public static class FileValidator
    implements IParameterValidator {
        @Override
        public void validate(String name, String value) throws ParameterException {
            try {
                File f = new File(value);
                f.getCanonicalPath();
            }
            catch (IOException e) {
                throw new ParameterException(String.format("Parameter %s \"%s\" does not seem to represent a valid file path: %s", name, value, e.getMessage()));
            }
        }
    }
}

