/******************************************************************************
**
** AirJack: 802.11b attack drivers for use with the AirJack set of tools...
**
**   Author:  Abaddon, abaddon@802.11ninja.net
**
**   Other Development Stuff:  Xx25,  xx25@leper.org
**
**   Copyright (c) 2002 Abaddon, All Rights Reserved (see license info below). 
**
********************
**
**    hfa384x.c:
**        HFA384x device specific code...
**
********************
**
** Legal/Credits:
**
**   While this code is unique, much of it is influenced by the Absolute 
**   Value Systems wlan-ng driver. 
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License
**   as published by the Free Software Foundation; either version 2
**   of the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this program; if not, write to the Free Software
**   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
********************
**
** $Id$
**
******************************************************************************/
#define __HFA384X_C__
#define __NO_VERSION__
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>
#include <pcmcia/bus_ops.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/types.h>

#include "airjack.h"


/*** Globals ***/

/***************/

/*** Prototypes ***/

void hfa384x_disable_interrupts (aj_info_t *);
void hfa384x_enable_interrupts (aj_info_t *);
void hfa384x_aux_enable (aj_info_t *);
void hfa384x_aux_disable (aj_info_t *);
void hfa384x_aux_read (aj_info_t *, __u32, __u16 *, ssize_t);
void hfa384x_aux_write (aj_info_t *, __u32, __u16 *, ssize_t);
ssize_t airjack_read_core (struct file *, char *, size_t, loff_t *);
void hfa384x_clear_status (aj_info_t *);
void hfa384x_busy_wait (aj_info_t *);
int hfa384x_command (aj_info_t *, __u16, __u16, __u16, __u16, cmd_resp_t *);
int hfa384x_txoff (aj_info_t *);
int hfa384x_txon (aj_info_t *);
int hfa384x_setup_bap (aj_info_t *, __u16, __u16, int);
int hfa384x_read_bap (aj_info_t *, __u16 *, size_t, int);
int hfa384x_write_bap (aj_info_t *, __u16 *, size_t, int);
int hfa384x_read_rid (aj_info_t *, __u16, void *, size_t, int);
int hfa384x_write_rid (aj_info_t *, __u16, void *, size_t, int);

/******************/


struct file_operations	airjack_operations = {
    llseek:		default_llseek,
    read:		airjack_read_core,
};


void hfa384x_disable_interrupts (aj_info_t *ai)
{
    
    /* a few times for good measure */
    OUT384x(ai, HFA384X_INTEN, 0);
    OUT384x(ai, HFA384X_INTEN, 0);
    OUT384x(ai, HFA384X_INTEN, 0);
}


void hfa384x_enable_interrupts (aj_info_t *ai)
{

    /* a few times for good measure */
    OUT384x(ai, HFA384X_INTEN, HFA384X_IRQ_MASK);
    OUT384x(ai, HFA384X_INTEN, HFA384X_IRQ_MASK);
    OUT384x(ai, HFA384X_INTEN, HFA384X_IRQ_MASK);
}


/*
 * enables the aux port on the hfa384x...
 *
 * infinate loops are scary i know...
 */
void hfa384x_aux_enable (aj_info_t *ai)
{
    __u16	control;


    hfa384x_busy_wait(ai);

    OUT384x(ai, HFA384X_PARAM0, HFA384X_AUXPW0);
    OUT384x(ai, HFA384X_PARAM1, HFA384X_AUXPW1);
    OUT384x(ai, HFA384X_PARAM2, HFA384X_AUXPW2);

    OUT384x(ai, HFA384X_CONTROL, HFA384X_CONTROL_AUX_DOENABLE);

    control = IN384x(ai, HFA384X_CONTROL);
    /* can you here me now? */
    while(((control & (BIT14|BIT15)) != HFA384X_CONTROL_AUX_ISENABLED) && (ai->link.state & DEV_PRESENT)) {
        udelay(10);

        /* how about now? */
        control = IN384x(ai, HFA384X_CONTROL);
    }
}


/*
 * disables the aux port on the hfa384x...
 */
void hfa384x_aux_disable (aj_info_t *ai)
{
    __u16	control;


    hfa384x_busy_wait(ai);

    OUT384x(ai, HFA384X_PARAM0, 0);
    OUT384x(ai, HFA384X_PARAM1, 0);
    OUT384x(ai, HFA384X_PARAM2, 0);

    OUT384x(ai, HFA384X_CONTROL, HFA384X_CONTROL_AUX_DODISABLE);

    control = IN384x(ai, HFA384X_CONTROL);
    /* can you here me now? */
    while(((control & (BIT14|BIT15)) != HFA384X_CONTROL_AUX_ISDISABLED) && (ai->link.state & DEV_PRESENT)) {
        udelay(10);

        /* how about now? */
        control = IN384x(ai, HFA384X_CONTROL);
    }
}


/*
 * reads from the AUX ports at the given offset for the given length...
 */
void hfa384x_aux_read (aj_info_t *ai, __u32 address, __u16 *buf, ssize_t buflen)
{
    __u16	auxpage = HFA384X_AUX_MKPAGE(address);
    __u16	auxoffset = HFA384X_AUX_MKOFF(address, HFA384X_AUX_CTL_EXTDS);	/* even address */

    OUT384x(ai, HFA384X_AUXPAGE, auxpage);
    OUT384x(ai, HFA384X_AUXOFFSET, auxoffset);

    udelay(5);

    while(buflen > 0) {
        *buf++ = IN384x(ai, HFA384X_AUXDATA);
        buflen -= 2;
        if(++auxoffset >= HFA384X_AUX_OFF_MAX) {
            auxoffset = 0;
            auxoffset = HFA384X_AUX_MKOFF(auxoffset, HFA384X_AUX_CTL_EXTDS);
            OUT384x(ai, HFA384X_AUXPAGE, ++auxpage);
            OUT384x(ai, HFA384X_AUXOFFSET, auxoffset);
        }
    }
    OUT384x(ai, HFA384X_AUXOFFSET, 0);
}


void hfa384x_aux_write (aj_info_t *ai, __u32 address, __u16 *buf, ssize_t buflen)
{
    __u16	auxpage = HFA384X_AUX_MKPAGE(address);
    __u16	auxoffset = HFA384X_AUX_MKOFF(address, HFA384X_AUX_CTL_EXTDS);


    OUT384x(ai, HFA384X_AUXPAGE, auxpage);
    OUT384x(ai, HFA384X_AUXOFFSET, auxoffset);

    udelay(5);

    while(buflen > 0) {
        OUT384x(ai, HFA384X_AUXDATA, *buf++);
        buflen -= 2;
        if(++auxoffset >= HFA384X_AUX_OFF_MAX) {
            auxoffset = 0;
            auxoffset = HFA384X_AUX_MKOFF(auxoffset, HFA384X_AUX_CTL_EXTDS);
            OUT384x(ai, HFA384X_AUXPAGE, ++auxpage);
            OUT384x(ai, HFA384X_AUXOFFSET, auxoffset);
        }
    }
}


/*
 * used for the proc interface to the HFA384x's core memory...
 */
ssize_t airjack_read_core (struct file *file, char *to_buf, size_t nbytes, loff_t *ppos)
{
    const struct proc_dir_entry	*pent = file->f_dentry->d_inode->u.generic_ip;
    aj_info_t					*ai = (aj_info_t *)pent->data;
    __u16						*buf;
    ssize_t						count;


    /* only network admins should be able to fuck with this */
    if(!capable(CAP_NET_ADMIN)) {
        goto no_perms;
    }

    if(!(ai->link.state & DEV_PRESENT)) {
        goto no_dev;
    }

    if(!access_ok(VERIFY_WRITE, to_buf, nbytes)) {
        goto invalid_args;
    }

    if(*ppos >= pent->size) {
        goto eof;
    }

    if(nbytes > pent->size) {
        nbytes = pent->size;
    }

    if((nbytes + *ppos) > pent->size) {
        nbytes = pent->size - *ppos;
    }

    MOD_INC_USE_COUNT;

    count = (nbytes + 1) & (~1);

    /* this can sleep so we do it before we hold the locks */
    if((buf = (__u16 *)kmalloc(count, GFP_KERNEL)) == NULL) {
        goto no_mem;
    }
    
    /* this is the part that we need to stop everything for a second */
    hfa384x_disable_interrupts(ai);

    /* this should really be a spinlock, but i dont really care */
    local_irq_disable();

    hfa384x_aux_enable(ai);

    hfa384x_aux_read(ai, *ppos, buf, count);
    
    hfa384x_aux_disable(ai);

    local_irq_enable();
    hfa384x_enable_interrupts(ai);

    __copy_to_user(to_buf, buf, nbytes);
    kfree(buf);

    *ppos += nbytes;
    MOD_DEC_USE_COUNT;
    return(nbytes);

eof:
    return(0);

no_mem:
    MOD_DEC_USE_COUNT;
    return(-ENOMEM);

invalid_args:
    return(-EINVAL);

no_dev:
    return(-ENODEV);

no_perms:
    return(-EPERM);
}


/*
 * clears the status register (acknowledges any pending interrupts)...
 */
void hfa384x_clear_status (aj_info_t *ai)
{
    __u16	status;

    DEBUG_PRINT("airjack_cs: hfa384x_clear_status: starting\n");
    status = IN384x(ai, HFA384X_EVSTAT);
    OUT384x(ai, HFA384X_EVACK, status);
    DEBUG_PRINT("airjack_cs: hfa384x_clear_status: leaving\n");
}


/*
 * waits for command.busy to not be set...
 *
 * XXXX - remove the timeout when you know your code is stable...
 */
void hfa384x_busy_wait (aj_info_t *dev)
{
    __u32	retries = 10000;	/* im paranoid about infinate loops in kernel mode */

    DEBUG_PRINT("airjack_cs: entered hfa384x_busy_wait().\n");

    while((__u16)((__u16)IN384x(dev, HFA384X_CMD) & (__u16)HFA384X_CMD_BUSY) && retries-- && (dev->link.state & DEV_PRESENT)) {
        udelay(10);
	}

    if(retries) {
        DEBUG_PRINT("airjack_cs: leaving hfa384x_busy_wait().\n");
        return;
    } else {
        printk(KERN_WARNING "airjack_cs: hfa384x_busy_wait: timeout while waiting for command busy status to clear, this is probably an internal bug in this driver " __FILE__ " line %u.\n", __LINE__);
        return;
    }
}


/*
 * this issues a command to the HFA394x...
 * the cmd_resp_t structure representing the command status and responce
 * to the command is put in the structure pointed to by resp...
 *
 * NOTE:
 *   At present this will not disable interrupts for you, so if this is 
 *   used in more than one context then you'll want to do apropriet locking
 */
int hfa384x_command (aj_info_t *dev, __u16 command, __u16 param0, __u16 param1, __u16 param2, cmd_resp_t *resp)
{
    long int	max_retries = CMD_MAX_RETRIES;
    __u16		status;

    DEBUG_PRINT("airjack_cs: entered hfa384x_command().\n");

    hfa384x_busy_wait(dev);

    OUT384x(dev, HFA384X_PARAM0, param0);
    OUT384x(dev, HFA384X_PARAM1, param1);
    OUT384x(dev, HFA384X_PARAM2, param2);
    OUT384x(dev, HFA384X_CMD, command);
    
    udelay(200);

    status = IN384x(dev, HFA384X_EVSTAT);
    while(max_retries-- && !(__u16)(status & (__u16)HFA384X_EVSTAT_CMD) && (dev->link.state & DEV_PRESENT)) {
        status = IN384x(dev, HFA384X_EVSTAT);
        udelay(100);
    }
    if(max_retries != -1) {
        /* fill in the responce structure */
        if(resp != NULL) {
            resp->status = IN384x(dev, HFA384X_STATUS);
            resp->resp0 =  IN384x(dev, HFA384X_RESP0);
            resp->resp1 =  IN384x(dev, HFA384X_RESP1);
            resp->resp2 =  IN384x(dev, HFA384X_RESP2);
        } else {
            IN384x(dev, HFA384X_STATUS);
            IN384x(dev, HFA384X_RESP0);
            IN384x(dev, HFA384X_RESP1);
            IN384x(dev, HFA384X_RESP2);
        }

        /* ack the processing of the status/respn */
        OUT384x(dev, HFA384X_EVACK, HFA384X_EVACK_CMD);

        DEBUG_PRINT("airjack_cs: leaving hfa384x_command().\n");
        return(SUCCESS);
    } else {
        printk(KERN_ERR "airjack_cs: hfa384x_command: Max tries exceeded waiting for command.\n");
        return(ERROR);
    }
}

void hfa384x_command_no_wait (aj_info_t *dev, __u16 command, __u16 param0, __u16 param1, __u16 param2)
{

    DEBUG_PRINT("airjack_cs: entered hfa384x_command().\n");

    hfa384x_busy_wait(dev);

    OUT384x(dev, HFA384X_PARAM0, param0);
    OUT384x(dev, HFA384X_PARAM1, param1);
    OUT384x(dev, HFA384X_PARAM2, param2);
    OUT384x(dev, HFA384X_CMD, command);
}

/*
 * prepares the BAP register for transfer of data...
 * must be called before hfa384x_read_read or hfa384x_write_bap...
 */
int hfa384x_setup_bap (aj_info_t *dev, __u16 rid, __u16 offset, int bap)
{
    __u16	retries = 10000;
    __u16	status;


    DEBUG_PRINT("airjack_cs: entered %s.\n", __func__);

    do {
        status = IN384x(dev, HFA384X_OFFSET0+bap);

        if(status & (__u16)HFA384X_OFFSET_BUSY) {
            udelay(2);
        } else if(status & (__u16)HFA384X_OFFSET_ERR) {
            printk(KERN_ERR "airjack_cs: %s: BAP error 0x%x %i\n", __func__, status, bap);
            return(ERROR);
        } else {
            break;
        }
    } while(--retries && (dev->link.state & DEV_PRESENT));

    retries = 10000;

    OUT384x(dev, HFA384X_SELECT0+bap, rid);
    udelay(10);

    OUT384x(dev, HFA384X_OFFSET0+bap, offset);

    do {
        status = IN384x(dev, HFA384X_OFFSET0+bap);

        if(status & (__u16)HFA384X_OFFSET_BUSY) {
            udelay(2);
        } else if(status & (__u16)HFA384X_OFFSET_ERR) {
            printk(KERN_ERR "airjack_cs: %s: BAP error 0x%x %i\n", __func__, status, bap);
            return(ERROR);
        } else {
            break;
        }
    } while(--retries && (dev->link.state & DEV_PRESENT));

    if(retries) {
        DEBUG_PRINT("airjack_cs: leaving %s.\n", __func__);
        return(SUCCESS);
    }
    if(!(dev->link.state & DEV_PRESENT)) {
        printk(KERN_ERR "airjack_cs: %s: %s: card removed while waiting for offset busy to clear.\n", __func__, dev->net_dev->name);
    } else {

        printk(KERN_ERR "airjack_cs: %s: BAP error: timeout while seting up BAP %i.\n", __func__, bap);
    }
    return(ERROR);
}


/*
 * this does a read from the bap register...you must call bap_setup 
 * first or we'll sick the dogs on ya...
 */
int hfa384x_read_bap (aj_info_t *dev, __u16 *buf, size_t len, int bap)
{
    size_t	i;

    DEBUG_PRINT("airjack_cs: entered %s\n", __func__);

    if(!(len%2)) {
        /* the hfa384x sucks so i have to read this all slow like */
        for(i=0;i < len;i += 2) {
            insw(dev->net_dev->base_addr+HFA384X_DATA0+bap, buf+(i>>1), 1);
        }
        DEBUG_PRINT("airjack_cs: leaving %s\n", __func__);
        return(SUCCESS);
    } else {
        printk(KERN_ERR "airjack_cs: %s: some jackass attempted to read an invalid amount of data (odd numbered) from the bap %u, NO...\n", __func__, (unsigned)len);
        return(ERROR);
    }
}


/*
 * this does a write to the bap register...you must call bap_setup
 * first or we'll sick the dogs on ya...
 */
int hfa384x_write_bap (aj_info_t *dev, __u16 *buf, size_t len, int bap)
{
    size_t	i;

    DEBUG_PRINT("airjack_cs: entered hfa384x_write_bap().\n");

    if(!(len%2)) {
        /* the hfa384x sucks so i have to read this all slow like */
        for(i=0;i < len;i += 2) {
            outsw(dev->net_dev->base_addr+HFA384X_DATA0+bap, buf+(i>>1), 1);
        }
        DEBUG_PRINT("airjack_cs: leaving hfa384x_write_bap().\n");
        return(SUCCESS);
    } else {
        printk(KERN_ERR "airjack_cs: write_bap: some jackass attempted to write an invalid amount of data (odd numbered) from the bap, NO...\n");
        return(ERROR);
    }
}


/*
 * this reads the given rid into buf from bap...
 *
 * XXXX - maybe change locks to make them rid specific 
 */
int hfa384x_read_rid (aj_info_t *dev, __u16 rid, void *buf, size_t len, int bap)
{
    int	rc;

    DEBUG_PRINT("airjack_cs: entered hfa384x_read_rid().\n");

    if(hfa384x_setup_bap(dev, rid, 0, bap) == SUCCESS) {
        hfa384x_read_bap(dev, (__u16 *)buf, 2, bap);
        if(len > 4) {
            rc = hfa384x_read_bap(dev, ((__u16 *)buf)+2, len - 2, bap);
            DEBUG_PRINT("airjack_cs: leaving hfa384x_read_rid().\n");
            return(rc);
        } else {
            printk(KERN_ERR "airjack_cs: hfa384x_read_rid: invalid RID length %u (too small).\n", len);
            return(ERROR);
        }
    } else {
        printk(KERN_ERR "airjack_cs: hfa384x_read_rid: error while trying to setup bap %i.\n", bap);
        return(ERROR);
    }
}


/*
 * this writes the given rid from buf to the bap...
 */
int hfa384x_write_rid (aj_info_t *dev, __u16 rid, void *buf, size_t len, int bap)
{
    DEBUG_PRINT("airjack_cs: entered hfa384x_write_rid().\n");
    
    if(hfa384x_setup_bap(dev, rid, 0, bap) == SUCCESS) {
        udelay(20);
        if(hfa384x_write_bap(dev, (__u16 *)buf, 4, bap) == SUCCESS) {
            udelay(20);
            if(hfa384x_setup_bap(dev, rid, 2*sizeof(__u16), bap) == SUCCESS) {
                udelay(20);
                if(hfa384x_write_bap(dev, ((__u16 *)buf) + 2, len - (2*sizeof(__u16)), bap) == SUCCESS) {
                    if(hfa384x_command(dev, HFA384X_CMDCODE_ACCESS|HFA384X_CMD_WRITE_SET(1), rid, 0, 0, NULL) == SUCCESS) {
                        DEBUG_PRINT("airjack_cs: leaving hfa384x_write_rid().\n");
                        return(SUCCESS);
                    }
                }
            }
        }
    }
    printk(KERN_ERR "airjack_cs: hfa384x_write_rid: error while trying to write to bap %i.\n", bap);
    return(ERROR);
}



