/******************************************************************************
**
** AirJack: 802.11b attack drivers for use with the AirJack set of tools...
**
**   Author:  Abaddon, abaddon@802.11ninja.net
**
**   Other Development Stuff:  Xx25,  xx25@leper.org
**
**   Copyright (c) 2002 Abaddon, All Rights Reserved (see license info below). 
**
********************
**
**    airjack.c:
**        All the kernel hooks and non-hardware specific parts of the code...
**
********************
**
** Legal/Credits:
**
**   While this code is unique, much of it is influenced by the Absolute 
**   Value Systems wlan-ng driver. 
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License
**   as published by the Free Software Foundation; either version 2
**   of the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this program; if not, write to the Free Software
**   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
********************
**
** $Id$
**
******************************************************************************/
#define __AIRJACK_C__
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/types.h>
#include <net/arp.h>

#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>
#include <pcmcia/bus_ops.h>

#include "airjack.h"
#include "80211.h"




#ifdef __SMP__
#error this is _not_ SMP safe code, sorry drive through...
#endif
#ifdef CONFIG_SMP
#error this is _not_ SMP safe code, sorry drive through...
#endif

#define INT_MODULE_PARM(n, v) static int n = v; MODULE_PARM(n, "i")
MODULE_AUTHOR("Abaddon, abaddon@802.11ninja.net");
MODULE_DESCRIPTION("AirJack");
MODULE_LICENSE("GPL");

/* Newer, simpler way of listing specific interrupts */
static int irq_list[4] = { -1 };
MODULE_PARM(irq_list, "1-4i");

static char	*mac_param = NULL;
MODULE_PARM(mac_param, "s");

/* The old way: bit map of interrupts to choose from */
/* This means pick from 15, 14, 12, 11, 10, 9, 7, 5, 4, and 3*/
//static unsigned int	irq_mask = 0xdeb0;
static unsigned int	irq_mask = 0xdeb8;
MODULE_PARM(irq_mask, "i");





/*** Prototypes ***/

static void flush_stale_links (void);
static void cs_error(client_handle_t, int, int);
static __inline__ void to_upper (char *);
static __inline__ int string_to_mac (char *, __u8 *);
static void stop_airjack_card (struct net_device *);
//static int reset_airjack_card (struct net_device *);
static int add_net_dev (struct net_device *);
static void del_net_dev (struct net_device *);
static int airjack_open (struct net_device *);
static int airjack_close (struct net_device *);
static int airjack_start_xmit(struct sk_buff *, struct net_device *);
struct net_device_stats *airjack_get_stats (struct net_device *);
static __inline__ void airjack_receive (aj_info_t *);
static void airjack_interrupt (int, void *, struct pt_regs *);
static int airjack_event (event_t, int, event_callback_args_t *);
static int airjack_ioctl (struct net_device *, struct ifreq *, int);
static __inline__ int setup_net_dev (aj_info_t *, __u16, int);
static __inline__ int enable_mac (aj_info_t *);
static __inline__ int finish_bootstrap (aj_info_t *);
static __inline__ int initial_config (aj_info_t *);
static void airjack_config (dev_link_t *);
static void airjack_release (unsigned long int);
static dev_link_t *airjack_attach (void);
static void airjack_detach (dev_link_t *);
static int init_airjack (void);
static void cleanup_airjack (void);

/******************/


/*** Globals ***/

/* XXXX - cfg_rid _needs_ to be moved into private data structure */
static dev_link_t	*dev_list = NULL;		/* list of devices */
static dev_info_t	dev_info = "airjack_cs";	/* name of the driver */
static __u8	mac_addr[6] = {'\x00', '\xde', '\xad', '\xc0', '\xde', '\x00'};	/* initial address of the card */
static struct net_device_list {
    struct net_device		*dev;
    struct net_device_list	*next;
} *airjack_devices = NULL;

/***************/

/*
 * stolen from David Hines...
 */
static void flush_stale_links (void)
{
    dev_link_t	*link, *next;

    DEBUG_PRINT("airjack_cs: flush_stale_links: starting.\n");
    for(link=dev_list; link != NULL;link=next) {
        next = link->next;
        if(link->state & DEV_STALE_LINK) {
            airjack_detach(link);
        }
    }
    DEBUG_PRINT("airjack_cs: flush_stale_links: leaving.\n");
}


/*
 * reports card services errors...
 */
static void cs_error (client_handle_t handle, int func, int ret)
{
    error_info_t	err = {func, ret};

    DEBUG_PRINT("airjack_cs: cs_error: starting\n");

    CardServices(ReportError, handle, &err);
    DEBUG_PRINT("airjack_cs: cs_error: leaving\n");
}


/*
 * converts all lower case letters in the the string to upper case
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static __inline__ void to_upper (char *s)
{
    char	*p;
    char	offset;
    
    DEBUG_PRINT("airjack_cs: to_upper: starting\n");

    offset = 'A' - 'a';
    for(p=s;*p != '\0';p++) {
        if(islower(*p)) {
            *p += offset;
        }
    }

    DEBUG_PRINT("airjack_cs: to_upper: leaving\n");
}


/*
 * converts a string to a mac address...
 * returns SUCCESS on success, and yep, ERROR on error...
 *
 * mac addresses must be of the form:
 * [0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}:[0-9A-F]{2}
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static __inline__ int string_to_mac (char *string, __u8 *mac)
{
    int				i;
    unsigned char	c, b, valid_hex[256];


    DEBUG_PRINT("airjack_cs: string_to_mac: starting\n");
    if(strlen(string) != 17) {
        DEBUG_PRINT("airjack_cs: string_to_mac: error\n");
        return(ERROR);
    }

    to_upper(string);

    /* now we'll strip out the ':''s */
    memmove(&string[14], &string[15], 3);
    memmove(&string[11], &string[12], 5);
    memmove(&string[8], &string[9], 7);
    memmove(&string[5], &string[6], 9);
    memmove(&string[2], &string[3], 11);

    memset(valid_hex, 0x10, sizeof(valid_hex));
    valid_hex['0'] = 0x00;
    valid_hex['1'] = 0x01;
    valid_hex['2'] = 0x02;
    valid_hex['3'] = 0x03;
    valid_hex['4'] = 0x04;
    valid_hex['5'] = 0x05;
    valid_hex['6'] = 0x06;
    valid_hex['7'] = 0x07;
    valid_hex['8'] = 0x08;
    valid_hex['9'] = 0x09;
    valid_hex['A'] = 0x0A;
    valid_hex['B'] = 0x0B;
    valid_hex['C'] = 0x0C;
    valid_hex['D'] = 0x0D;
    valid_hex['E'] = 0x0E;
    valid_hex['F'] = 0x0F;

    for(i=0;i < 6;i++) {
        if((c = valid_hex[(unsigned char)string[i*2]]) == 0x10) {
            DEBUG_PRINT("airjack_cs: string_to_mac: error\n");
            return(ERROR);
        }
        if((b = valid_hex[(unsigned char)string[i*2 + 1]]) == 0x10) {
            DEBUG_PRINT("airjack_cs: string_to_mac: error\n");
            return(ERROR);
        }
        mac[i] = (__u8)((c*16) + b);
    }

    DEBUG_PRINT("airjack_cs: string_to_mac: leaving\n");
    return(SUCCESS);
}


/*
 * stops the airjack network device and halts the hfa384x...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static void stop_airjack_card (struct net_device *net_dev)
{
    aj_info_t	*ai = (aj_info_t *)net_dev->priv;

    DEBUG_PRINT("airjack_cs: stop_airjack_card: stoping aironet card.\n");
    flush_scheduled_tasks();

    hfa384x_disable_interrupts(ai);
    hfa384x_command(ai, HFA384X_CMDCODE_DISABLE, 0, 0, 0, NULL);

    if(ai->irq_set) {
        free_irq(net_dev->irq, net_dev);
        ai->irq_set = 0;
    }
    del_net_dev(net_dev);

    if(ai->registered) {
        unregister_netdev(net_dev);
        netdev_finish_unregister(net_dev);
        ai->registered = 0;
    }
    DEBUG_PRINT("airjack_cs: stop_airjack_card: aironet card should be stopped.\n");
}


/*
 * allocates a transmit Fid...
 * on success an fid is returned, on failure zero...
 */
static __inline__ __u16 allocate_txfid (aj_info_t *ai)
{
    cmd_resp_t			resp;
    __u16				fid;


    memset(&resp, 0, sizeof(resp));

    if(hfa384x_command(ai, HFA384X_CMDCODE_ALLOC, 1700, 0, 0, &resp) != SUCCESS) {
        goto error;
    }

    if((resp.status & 0xFF00) != 0) {
        goto error;
    }

    /* Ben Reed says this will only loop like 4 times, lets hope he's right */
    while(((__u16)((__u16)IN384x(ai, HFA384X_EVSTAT) & (__u16)HFA384X_EVSTAT_ALLOC) == 0) && (ai->link.state & DEV_PRESENT));

    fid = IN384x(ai, HFA384X_ALLOCFID);
    OUT384x(ai, HFA384X_EVACK, HFA384X_EVACK_ALLOC);

    return(fid);

error:
    return(0);
}


/*
 * resets the hfa384x...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
/*
static int reset_airjack_card (struct net_device *net_dev)
{

    DEBUG_PRINT("airjack_cs: reset_airjack_card: starting reset\n");


    DEBUG_PRINT("airjack_cs: reset_airjack_card: leaving\n");
    return(SUCCESS);
}
*/


/*
 * adds a net device to the list...
 */
static int add_net_dev (struct net_device *net_dev)
{
    struct net_device_list	*node;

    DEBUG_PRINT("airjack_cs: add_net_dev: starting.\n");
    if((node = (struct net_device_list *)kmalloc(sizeof(struct net_device_list), GFP_KERNEL)) == NULL) {
        DEBUG_PRINT("airjack_cs: add_net_dev: error.\n");
        return(-ENOMEM);
    }

    node->dev = net_dev;
    node->next = airjack_devices;
    airjack_devices = node;

    DEBUG_PRINT("airjack_cs: add_net_dev: leaving.\n");
    return(SUCCESS);
}


/*
 * removes a device from the list...
 */
static void del_net_dev (struct net_device *net_dev)
{
    struct net_device_list	**p, *free;

    DEBUG_PRINT("airjack_cs: del_net_dev: starting.\n");
    p = &airjack_devices;
    while(*p && ((*p)->dev != net_dev)) {
        p = &(*p)->next;
    }
    if(*p && ((*p)->dev == net_dev)) { 
        free = *p;
        *p = (*p)->next;
        kfree(free);
    }
    DEBUG_PRINT("airjack_cs: del_net_dev: leaving.\n");
}


/*
 * starts the network device (enables interrupts and starts the queue...
 */
static int airjack_open (struct net_device *net_dev)
{
    aj_info_t	*ai = (aj_info_t *)net_dev->priv;

    DEBUG_PRINT("airjack_cs: %s: starting\n", __func__);
    //hfa384x_clear_status(ai);
    netif_start_queue(net_dev);
    hfa384x_enable_interrupts(ai);
    DEBUG_PRINT("airjack_cs: %s: leaving\n", __func__);
    return(0);
}


/*
 * this shuts down the net device, disables interrupts and stops the queue...
 */
static int airjack_close (struct net_device *net_dev)
{
    aj_info_t	*ai = (aj_info_t *)net_dev->priv;

    DEBUG_PRINT("airjack_cs: %s: starting\n", __func__);
    hfa384x_disable_interrupts(ai);
    //hfa384x_clear_status(ai);
    netif_stop_queue(net_dev);
    DEBUG_PRINT("airjack_cs: %s: leaving\n", __func__);
    return(0);
}


/*
 * handles transmiting of frames...
 *
 *
 ****
 * Note:
 *   This is rather lazy about endianness, its assuming little endian, this needs to be 
 *   fixed if anyone cares...
 *
 ****
 * Call Context:
 *   Interrupt Context
 *
 */
static int airjack_start_xmit(struct sk_buff *skb, struct net_device *net_dev)
{
    aj_info_t			*ai = (aj_info_t *)(net_dev->priv);
    struct tx_control	tx_ctl;
    __u16				fid;


    if(skb == NULL) {
        goto null_skb;
    }

    if(skb->data == NULL) {
        goto null_data;
    }
    
    if(skb->len < sizeof(struct a3_80211)) {
        goto dead_frame;
    }


/*
    if(ai->tbusy) {
        goto requeue_error;
    }
*/

    memset(&tx_ctl, 0, sizeof(tx_ctl));

    /* copy the first sizeof(struct a3_80211) bytes to the header section in the tx descriptor */
    memcpy(&(tx_ctl.tc_fc), skb->data, sizeof(struct a3_80211));
    tx_ctl.tc_data_len = skb->len - sizeof(struct a3_80211);

    tx_ctl.tc_tx_ctl = HFA384X_TX_CFPOLL_SET(1) | 0x08 | HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0) | HFA384X_TX_RETRYSTRAT_SET(3);
    //tx_ctl.tc_tx_ctl = 0x08 | HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0);

    /* set the tc_dest_addr and tc_src_addr for data frames */
/*
    if(((struct a3_80211 *)skb->data)->mh_type == FC_TYPE_DATA) {
        if(((struct a3_80211 *)skb->data)->mh_from_ds) {
            memcpy(tx_ctl.tc_dest_addr, ((struct a3_80211 *)skb->data)->mh_mac1, 6);
            memcpy(tx_ctl.tc_src_addr, ((struct a3_80211 *)skb->data)->mh_mac3, 6);
        } else {
            memcpy(tx_ctl.tc_dest_addr, ((struct a3_80211 *)skb->data)->mh_mac3, 6);
            memcpy(tx_ctl.tc_src_addr, ((struct a3_80211 *)skb->data)->mh_mac2, 6);
        }
        //tx_ctl.tc_tx_ctl = HFA384X_TX_CFPOLL_SET(1) | HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0) | HFA384X_TX_STRUCTYPE_SET(2);
        //tx_ctl.tc_tx_ctl = HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0) | ;
        tx_ctl.tc_tx_ctl = 0x08 | HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0) | HFA384X_TX_RETRYSTRAT_SET(3);
        //tx_ctl.tc_data_length = htons(skb->len - sizeof(struct a3_80211));
    } else {
        tx_ctl.tc_tx_ctl = HFA384X_TX_CFPOLL_SET(1) | 0x08 | HFA384X_TX_TXEX_SET(1) | HFA384X_TX_TXOK_SET(1) | HFA384X_TX_MACPORT_SET(0) | HFA384X_TX_RETRYSTRAT_SET(3);
    }
*/

    /* not the best way to do this i know, but it works */
    if((fid = allocate_txfid(ai)) == 0) {
        goto requeue_error;
    }
    udelay(20);

    if(hfa384x_setup_bap(ai, fid, 0x00, HFA384X_BAP1) != SUCCESS) {
        goto requeue_error;
    }

    /* write the tx_control header to the fid buffer */
    if(hfa384x_write_bap(ai, (__u16 *)&tx_ctl, sizeof(tx_ctl), HFA384X_BAP1) != SUCCESS) {
        goto requeue_error;
    }

    /* write the payload (if any) to the bap */
    if(tx_ctl.tc_data_len != 0) {
        /* i think we have to write 16 bytes at a time, no less on this chip */
        if(!((skb->len - sizeof(struct a3_80211)) % sizeof(__u16))) {
            if(hfa384x_write_bap(ai, (__u16 *)((__u8 *)skb->data + sizeof(struct a3_80211)), skb->len - sizeof(struct a3_80211), HFA384X_BAP1) != SUCCESS) {
                goto requeue_error;
            }
        } else {
            union {
                __u16	u16;
                __u8	u8[2];
            } extra_byte;

            if(hfa384x_write_bap(ai, (__u16 *)((__u8 *)skb->data + sizeof(struct a3_80211)), skb->len - sizeof(struct a3_80211) - 1, HFA384X_BAP1) != SUCCESS) {
                goto requeue_error;
            }
            extra_byte.u8[0] = *(__u8 *)(skb->data + (skb->len - 1));
            extra_byte.u8[1] = 0;
            if(hfa384x_write_bap(ai, &extra_byte.u16, sizeof(extra_byte), HFA384X_BAP1) != SUCCESS) {
                goto requeue_error;
            }
        }
    }

    if(hfa384x_command(ai, HFA384X_CMDCODE_TX|HFA384X_CMD_RECL_SET(0)|HFA384X_CMD_QOS_SET(0), fid, 0, 0, NULL) != SUCCESS) {
        goto requeue_error;
    }

//    netif_stop_queue(ai->net_dev);
    return(SUCCESS);

requeue_error:
printk("requeueing frame\n");
    if(ai->tbusy == 3) {
        ai->tbusy = 0;
    } else {
        ai->tbusy++;
    }
    ai->stats.tx_errors++;
    return(1);
dead_frame:
    ai->stats.tx_dropped++;
    ai->stats.tx_errors++;
    dev_kfree_skb(skb);
    return(0);

null_skb:
    printk(KERN_ALERT "airjack_cs: Man the life boats, women and children firsts, the kernel just passed me a NULL skb\n");
    return(0);

null_data:
    printk(KERN_ALERT "airjack_cs: Man the life boats, women and children firsts, the kernel just passed me a NULL skb->data\n");
    return(0);
}


/*
 * returns net device statistics...
 */
struct net_device_stats *airjack_get_stats (struct net_device *net_dev)
{
    aj_info_t	*ai = (aj_info_t *)net_dev->priv;

    return(&ai->stats);
}


/*
 * receives frames from the card, processes them and passes them
 * to the higher layers...
 * 
 * Note:
 *  if you dont compile this with -O2 on gcc then these goto's will not help
 *  but rather hurt...
 *
 ****
 * Call Context:
 *   Hardware Interrupt Context
 *
 */
static __inline__ void airjack_receive (aj_info_t *ai)
{
    struct sk_buff		*skb = NULL;
    struct llc_hdr		*llc;
    __u16				*buf;
    __u8				*dst_mac;
    __u16				hdrlen, fid;
    struct {
        __u16	status;
        __u32	time;
        __u8	silence;
        __u8	signal;
        __u8	rate;
        __u8	rx_flow;
        __u16	reserved1;
        __u16	reserved2;
        __u16	fc_version:2;
        __u16	fc_type:2;
        __u16	fc_subtype:4;
        __u16	fc_to_ds:1;
        __u16	fc_from_ds:1;
        __u16	fc_more_frag:1;
        __u16	fc_retry:1;
        __u16	fc_pwr_man:1;
        __u16	fc_more_data:1;
        __u16	fc_wep:1;
        __u16	fc_order:1;
        __u16	duration_id;
        __u8	address1[6];
        __u8	address2[6];
        __u8	address3[6];
        __u16	sequence_control;
        __u8	address4[6];
        __u16	data_len;
        __u8	dest_addr[6];
        __u8	src_addr[6];
        __u16	data_length;
    } __attribute__ ((packed)) fid_hdr;
    

    /* get the rxFid */
    fid = IN384x(ai, HFA384X_RXFID);
    
    if(hfa384x_setup_bap(ai, fid, 0x00, HFA384X_BAP0) != SUCCESS) {
        printk(KERN_DEBUG "error while setting up bap");
        goto rx_error;
    }

    /* read in the rx desc and headers */
    hfa384x_read_bap(ai, (__u16 *)&fid_hdr, sizeof(fid_hdr), HFA384X_BAP0);

    if(fid_hdr.status & HFA384X_RXSTATUS_FCSERR) {
        goto crc_error;
    }

    /* wow, is it me or does this sort of resemble an ioccc entry?, mmm, bad code */
    switch(fid_hdr.fc_type) {
    case FC_TYPE_DATA:
        hdrlen = sizeof(struct a3_80211);
        switch((fid_hdr.fc_to_ds << 1) | (fid_hdr.fc_from_ds)) {
        case 0:
        case 1:
            /* these are offsets from 0, we'll ajust this in a second */
            dst_mac = (__u8 *)4;
            break;
        case 3:
            /* nothing like clearly written code ;) */
            hdrlen = sizeof(struct a4_80211);
        case 2:         /* Fall Through */
        default:
            dst_mac = (__u8 *)16;
            break;
        }
        break;
    case FC_TYPE_MGT:
        dst_mac = (__u8 *)4;
        hdrlen = sizeof(struct a3_80211);
        break;
    case FC_TYPE_CTL:
        dst_mac = (__u8 *)4;
        if(!((fid_hdr.fc_subtype == CTL_ACK) || (fid_hdr.fc_subtype == CTL_CTS))) {
            hdrlen = sizeof(struct a2_80211);
        } else {
            hdrlen = sizeof(struct a1_80211);
        }
        break;
    default:	/* reserved type */
        dst_mac = (__u8 *)4;
        hdrlen = 24;
    }

    if((skb = dev_alloc_skb((fid_hdr.data_len + hdrlen + 1) & (~1))) == NULL) {
        goto rx_dropped;
    }

    buf = (__u16 *)(skb->data);

    /* 14 is the offset in the rx info struct */
    memcpy(buf, (__u8 *)(&fid_hdr) + 14, hdrlen);
    dst_mac = (__u8 *)((__u32)buf + (__u32)dst_mac);

    /* read in the payload */
    if(fid_hdr.data_len) {
        hfa384x_read_bap(ai, buf + (hdrlen>>1), (fid_hdr.data_len + 1) & (~1), HFA384X_BAP0);
    }

    ai->net_dev->last_rx = jiffies;
    skb->dev = ai->net_dev;
    skb->ip_summed = CHECKSUM_NONE;
    skb->mac.raw = skb->data;
    skb_put(skb, fid_hdr.data_len + hdrlen);
    llc = (struct llc_hdr *)(((__u8*)skb->data)+hdrlen);

    skb->protocol = __constant_htons(ETH_P_802_2);
    skb->pkt_type = PACKET_OTHERHOST;	/* we'll start with this assumption */

    /* we have to check to see if we're pretending to be the person this is to */
#if 0
    if(!((!memcmp(dst_mac, ai->net_dev->dev_addr, 6)) || (dst_mac[0] & 0x01)) && (fid_hdr.fc_type == FC_TYPE_DATA)) {
        skb->protocol = __constant_htons(ETH_P_802_2);
        //skb->pkt_type = PACKET_HOST;
        skb->pkt_type = PACKET_OTHERHOST;
    } else {
        if(!(fid_hdr.status & (HFA384X_RXSTATUS_FCSERR))) {

            skb->protocol = htons(llc->type);
            /* we'll pretend this one is ours and pass it to normal sockets */
            if((dst_mac[0] & 0x01) & 0x01) {
                if(!memcmp(dst_mac, "\xff\xff\xff\xff\xff\xff", 6)) {
                    skb->pkt_type = PACKET_BROADCAST;
                } else {
                    skb->pkt_type = PACKET_MULTICAST;
                }
                ai->stats.multicast++;
            } 
        } else {
            skb->protocol = __constant_htons(ETH_P_802_2);
            skb->pkt_type = PACKET_OTHERHOST;
            //skb->pkt_type = PACKET_HOST;
        }
    }
#endif

    /* update statistics */
    ai->stats.rx_packets++;
    ai->stats.rx_bytes += skb->len;
    netif_rx(skb);
    return;

crc_error:
    ai->stats.rx_crc_errors++;
rx_error:
    ai->stats.rx_errors++;
    return;

rx_dropped:
    printk(KERN_DEBUG "airjack_cs: airjack_receive: rx dropped.\n");
    ai->stats.rx_dropped++;
    return;
}


/*
 * interrupt handler...we talk to the card then schedule the apropriate
 * bottom half to process and do its thing...
 *
 *
 ****
 * Call Context:
 *   Hardware Interrupt Context
 *
 */
static void airjack_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
    struct net_device	*net_dev = (struct net_device *)dev_id;
    aj_info_t			*ai = net_dev->priv;
    __u16				status;


    hfa384x_disable_interrupts(ai);
    status = IN384x(ai, HFA384X_EVSTAT);
    if(netif_device_present(net_dev)) {
        if((__u16)(((__u16)(status)) & HFA384X_EVSTAT_RX)) {
            airjack_receive(ai);
            OUT384x(ai, HFA384X_EVACK, ((__u16)(((__u16)(1)) << 0)));
        }

        if((__u16)(((__u16)(status)) & HFA384X_EVSTAT_TX)) {
            ai->stats.tx_packets++;

            if(ai->tbusy) {
                ai->tbusy--;
            }

            /* acknowledge the interrupt */
            OUT384x(ai, HFA384X_EVACK, ((__u16)(((__u16)(1)) << 1)));
            netif_wake_queue(net_dev);
        } 

        if((__u16)(((__u16)(status)) & HFA384X_EVSTAT_TXEXC)) {
            /* XXXX - check for status 1 (requeue error) and ignore it */
            ai->stats.tx_errors++;

            if(ai->tbusy) {
                ai->tbusy--;
            }

            /* acknowledge the interrupt */
            OUT384x(ai, HFA384X_EVACK, ((__u16)(((__u16)(1)) << 2)));
            netif_wake_queue(net_dev);
        } 

        hfa384x_enable_interrupts(ai);
        return;
    } else {
        if(ai->link.state & DEV_PRESENT) {
            printk("\n");
            printk(KERN_WARNING "airjack_cs: %s: interrupt from stoped card %s !!\n", __func__, net_dev->name);
        }

        /* acknowledge the interrupt anyways */
        OUT384x(ai, HFA384X_EVACK, status);
        hfa384x_enable_interrupts(ai);
        printk(KERN_NOTICE "airjack_cs: %s: device not present\n", __func__);
        return;
    }
}


/*
 * this is the Card Services event handler...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static int airjack_event (event_t event, int priority, event_callback_args_t *args)
{
    dev_link_t	*link = (dev_link_t *)args->client_data;
    aj_info_t	*ai = (aj_info_t *)link->priv;
    
    DEBUG_PRINT("airjack_cs: airjack_event: starting.\n");

    switch(event) {
    case CS_EVENT_CARD_REMOVAL:
        DEBUG_PRINT("airjack_cs: airjack_event: got a card removal event.\n");
        link->state &= ~DEV_PRESENT;
        if (link->state & DEV_CONFIG) {
            DEBUG_PRINT("airjack_cs: airjack_event: defering release while card is busy.\n");
            netif_device_detach(ai->net_dev);
            mod_timer(&link->release, jiffies + HZ/20);
        }
        break;
    case CS_EVENT_CARD_INSERTION:
        DEBUG_PRINT("airjack_cs: airjack_event: got a card insertion event.\n");
        link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
        airjack_config(link);
        break;
    case CS_EVENT_PM_SUSPEND:
        DEBUG_PRINT("airjack_cs: airjack_event: got a card pm suspend event.\n");
        link->state |= DEV_SUSPEND;
        /* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
        DEBUG_PRINT("airjack_cs: airjack_event: got a card reset physical event.\n");
        if (link->state & DEV_CONFIG) {
            netif_device_detach(ai->net_dev);
            CardServices(ReleaseConfiguration, link->handle);
        }
        break;
    case CS_EVENT_PM_RESUME:
        DEBUG_PRINT("airjack_cs: airjack_event: got a card pm resume event.\n");
        link->state &= ~DEV_SUSPEND;
        /* Fall through... */
    case CS_EVENT_CARD_RESET:
        break;
    }
    DEBUG_PRINT("airjack_cs: airjack_event: leaving.\n");
    return(SUCCESS);
}


/*
 * handles device configuration through ioctl calls...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static int airjack_ioctl (struct net_device *net_dev, struct ifreq *rq, int cmd)
{
    aj_info_t			*ai = (aj_info_t *)(net_dev->priv);
    struct aj_config	config;

    /* only net admins can use this interface */
    if(!capable(CAP_NET_ADMIN)) {
        return(-EPERM);
    }

    switch(cmd) {
    case SIOCAJSMODE:
        {
            struct {
                __u16	len;
                __u16	rid;
                __u16	word;
            } rid;
            struct {
                __u16	len;
                __u16	rid;
                __u8	mac[6];
            } mac_rid;
            struct {
                __u16	len;
                __u16	rid;
                __u8	essid[34];
            } essid_rid;

            local_irq_disable();

            if(!access_ok(VERIFY_READ, rq->ifr_data, sizeof(struct aj_config))) {
                local_irq_enable();
                return(-EFAULT);
            }
            /* im not sure if this can sleep, if so i need to move this out of the critical section */
            __copy_from_user(&config, rq->ifr_data, sizeof(struct aj_config));

            /* verify what we've been given */
            if((config.channel < 1) || (config.channel > 14) || (config.essid[0] > 32) || (config.essid[0] == 0)) {
                local_irq_enable();
                return(-EINVAL);
            }

            if(hfa384x_command(ai, HFA384X_CMDCODE_DISABLE|HFA384X_TX_MACPORT_SET(0), 0, 0, 0, NULL) != SUCCESS) {
                local_irq_enable();
                printk(KERN_ERR "airjack_cs: %s: failed to disable mac.\n", __func__);
                return(ERROR);
            }
//            hfa384x_command_no_wait(ai, HFA384X_CMDCODE_DISABLE|HFA384X_TX_MACPORT_SET(0), 0, 0, 0);
            memset(&rid, 0, sizeof(rid));

#if 0
            rid.word = 1;
            rid.len = sizeof(rid)/2 - 1;
            rid.rid = HFA384X_RID_CREATEIBSS;
            if(hfa384x_write_rid(ai, HFA384X_RID_CREATEIBSS, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
                local_irq_enable();
                printk(KERN_ERR "airjack_cs: %s: failed to set create ibss mode.\n", __func__);
                return(ERROR);
            }

            rid.word = HFA384X_DISABLE_RXCRYPT;
            rid.len = sizeof(rid)/2 - 1;
            rid.rid = HFA384X_RID_CNFWEPFLAGS;
            /* turn off encryption */
            if(hfa384x_write_rid(ai, HFA384X_RID_CNFWEPFLAGS, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
                local_irq_enable();
                printk(KERN_ERR "airjack_cs: %s: failed to set wep flags.\n", __func__);
                return(ERROR);
            }

            rid.word = 100;
            rid.rid = HFA384X_RID_CNFBEACONINT;
            if(hfa384x_write_rid(ai, HFA384X_RID_CNFBEACONINT, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
                local_irq_enable();
                printk(KERN_ERR "airjack_cs: %s: failed to set beacon interval.\n", __func__);
                return(ERROR);
            }
#endif

            /* set the channel */
            if(config.channel != ai->channel) {
                rid.len = sizeof(rid)/2 - 1;
                rid.rid = HFA384X_RID_CNFOWNCHANNEL;
                rid.word = config.channel;
                if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNCHANNEL, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
                    enable_mac(ai);
                    local_irq_enable();
                    printk(KERN_ERR "airjack_cs: %s: failed to set channel.\n", __func__);
                    return(-EIO);
                }
                ai->channel = config.channel;
            }

            if(config.mode != ai->port_mode) {
                rid.word = config.mode;
                rid.len = sizeof(rid)/2 - 1;
                rid.rid = HFA384X_RID_CNFPORTTYPE;
                if(hfa384x_write_rid(ai, HFA384X_RID_CNFPORTTYPE, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
                    enable_mac(ai);
                    local_irq_enable();
                    printk(KERN_ERR "airjack_cs: %s: failed to change port type.\n", __func__);
                    return(-EIO);
                }
                ai->port_mode = config.mode;
            }

            /* set the cards mac */
            if(memcmp(ai->mac, config.ownmac, 6)) {
                memcpy(mac_rid.mac, config.ownmac, sizeof(mac_rid.mac));
                mac_rid.len = sizeof(mac_rid)/2 - 1;
                mac_rid.rid = HFA384X_RID_CNFOWNMACADDR;
                if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNMACADDR, &mac_rid, sizeof(mac_rid), HFA384X_BAP1) != SUCCESS) {
                    enable_mac(ai);
                    local_irq_enable();
                    printk(KERN_ERR "airjack_cs: %s: failed to turn off encryption.\n", __func__);
                    return(-EIO);
                }
                memcpy(ai->mac, config.ownmac, sizeof(mac_rid.mac));
                memcpy(ai->net_dev->dev_addr, config.ownmac, 6);
            }

            if(memcmp(ai->essid, config.essid, sizeof(config.essid))) {
                memset(&essid_rid, 0, sizeof(essid_rid));
                memcpy(essid_rid.essid + 2, config.essid + 1, config.essid[0]);
                essid_rid.essid[0] = config.essid[0];
                essid_rid.len = sizeof(essid_rid)/2 - 1;
                essid_rid.rid = HFA384X_RID_CNFOWNSSID;
                if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNSSID, &essid_rid, sizeof(essid_rid), HFA384X_BAP1) != SUCCESS) {
                    enable_mac(ai);
                    local_irq_enable();
                    printk(KERN_ERR "airjack_cs: %s: failed to turn off encryption.\n", __func__);
                    return(-EIO);
                }
                essid_rid.rid = HFA384X_RID_CNFDESIREDSSID;
                if(hfa384x_write_rid(ai, HFA384X_RID_CNFDESIREDSSID, &essid_rid, sizeof(essid_rid), HFA384X_BAP1) != SUCCESS) {
                    enable_mac(ai);
                    local_irq_enable();
                    printk(KERN_ERR "airjack_cs: %s: failed to turn off encryption.\n", __func__);
                    return(-EIO);
                }
                memcpy(ai->essid, config.essid, sizeof(ai->essid));
            }

            if(enable_mac(ai) != SUCCESS) {
                local_irq_enable();
                printk(KERN_ERR "airjack_cs: %s: failed to enable mac.\n", __func__);
                return(-EIO);
            }

            if(config.monitor != ai->monitor) {
                if(config.monitor) {
                    hfa384x_command(ai, HFA384X_CMDCODE_MONITOR|(HFA384X_MONITOR_ENABLE << 8), 0, 0, 0, NULL);
                } else {
                    hfa384x_command(ai, HFA384X_CMDCODE_MONITOR|(HFA384X_MONITOR_DISABLE << 8), 0, 0, 0, NULL);
                }
                ai->monitor = config.monitor;
            }

            local_irq_enable();
        }
        break;
    case SIOCAJGMODE:
        local_irq_disable();
        if(!access_ok(VERIFY_WRITE, rq->ifr_data, sizeof(struct aj_config))) {
            local_irq_enable();
            return(-EINVAL);
        }
        config.channel = ai->channel;
        config.mode = ai->port_mode;
        config.monitor = ai->monitor;
        memcpy(config.ownmac, ai->mac, 6);
        memcpy(config.essid, ai->essid, 33);
        /* im not sure if this can sleep, if so i need to move this out of the critical section */
        __copy_to_user(rq->ifr_data, &config, sizeof(struct aj_config));
        local_irq_enable();
        break;
    default:
        return(-ENOSYS);
    }
    return(SUCCESS);
}


/*
 * sets up the net_dev structure...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static __inline__ int setup_net_dev (aj_info_t *ai, __u16 irq, int port)
{
    long	flags;
    int		rc;

    /* get the dev_base_lock */
    write_lock_irqsave(&dev_base_lock, flags);

    DEBUG_PRINT("airjack_cs: setup_net_dev: starting\n");
    if((ai->net_dev = dev_alloc(AIRJACK_DEV_NAME "%d", &rc)) == NULL) {
        write_unlock_irqrestore(&dev_base_lock, flags);
        printk(KERN_ERR "airjack_cs: setup_net_dev: failed to allocate net device, error code %u.\n", -1*rc);
        return(ERROR);
    }

    /* ifindex is set by register_netdevice */
    write_unlock_irqrestore(&dev_base_lock, flags);

    memcpy(ai->net_dev->dev_addr, mac_addr, 6);
    ai->net_dev->addr_len = 6;
    memcpy(ai->net_dev->broadcast, "\xff\xff\xff\xff\xff\xff", 6);

    ai->net_dev->mtu = 1600;	/* no soup for you! */
    ai->net_dev->tx_queue_len = 1000;
    ai->net_dev->type = ARPHRD_IEEE80211;
    ai->net_dev->hard_header_len = sizeof(struct a4_80211) + sizeof(struct llc_hdr);
    
    if(add_net_dev(ai->net_dev) != SUCCESS) {
        printk(KERN_ERR "airjack_cs: setup_net_dev: failed to add network device.\n");
        /* do i need to free the device name seperately? */
        kfree(ai->net_dev);
        return(ERROR);
    }

    /* set stuff */
    ai->net_dev->priv = ai;
    ai->net_dev->irq = irq;
    ai->net_dev->base_addr = port;

    /* mark it as not registered yet... */
	ai->registered = 0;

    /* set callbacks */
    ai->net_dev->hard_start_xmit = &airjack_start_xmit;
    ai->net_dev->get_stats = &airjack_get_stats;
    ai->net_dev->do_ioctl = &airjack_ioctl;
    ai->net_dev->open = &airjack_open;
    ai->net_dev->stop = &airjack_close;

    DEBUG_PRINT("airjack_cs: setup_net_dev: leaving\n");
    return(SUCCESS);
}


/*
 * enables the mac in receive only mode...
 */
static __inline__ int enable_mac (aj_info_t *ai)
{
    cmd_resp_t	resp;

    DEBUG_PRINT("airjack_cs: %s: starting\n", __func__);

    memset(&resp, 0, sizeof(resp));

    if(hfa384x_command(ai, HFA384X_CMDCODE_ENABLE|HFA384X_TX_MACPORT_SET(0), 0, 0, 0, &resp) != SUCCESS) {
        printk(KERN_ERR "airjack_cs: %s: error while enabling MAC processor port 0, status = %u resp0 = %u resp1 = %u resp2 = %u\n", __func__, resp.status, resp.resp0, resp.resp1, resp.resp2);
        return(ERROR);
    }

    DEBUG_PRINT("airjack_cs: %s: leaving\n", __func__);
    return(SUCCESS);
}


/*
 * finishes the initial hfa384x bootstrap process and 
 * puts the card in a state ready to accept initial
 * configuration...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static __inline__ int finish_bootstrap (aj_info_t *ai)
{
    cmd_resp_t	resp;

    DEBUG_PRINT("airjack_cs: %s: starting\n", __func__);

    memset(&resp, 0, sizeof(resp));
    if(enable_mac(ai) != SUCCESS) {
        printk(KERN_ERR "airjack_cs: %s: failed to enable mac.\n", __func__);
        return(ERROR);
    }

    hfa384x_command(ai, HFA384X_CMDCODE_MONITOR|(HFA384X_MONITOR_ENABLE << 8), 0, 0, 0, NULL);
    ai->monitor = 1;

    DEBUG_PRINT("airjack_cs: %s: leaving\n", __func__);
    return(SUCCESS);
}


/*
 * sets the initial card configuration parameters...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static __inline__ int initial_config (aj_info_t *ai)
{
    cmd_resp_t	resp;
    struct {
        __u16	len;
        __u16	rid;
        __u16	word;
    } rid;
    struct {
        __u16	len;
        __u16	rid;
        __u8	mac[6];
    } mac_rid;
    struct {
        __u16	len;
        __u16	rid;
        __u8	essid[34];
    } essid_rid;


    DEBUG_PRINT("airjack_cs: %s: starting.\n", __func__);

    hfa384x_disable_interrupts(ai);

    local_irq_disable();

    memset(&resp, 0, sizeof(resp));

    /* we need to init the mac */
    if(hfa384x_command(ai, HFA384X_CMDCODE_INIT, 0, 0, 0, &resp) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: init mac command failed, status = %u resp0 = %u resp1 = %u resp2 = %u.\n", __func__, resp.status, resp.resp0, resp.resp1, resp.resp2);
        return(ERROR);
    }

    if(hfa384x_command(ai, HFA384X_CMDCODE_DISABLE, 0, 0, 0, &resp) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: error while disabling MAC processor port 0, status = %u resp0 = %u resp1 = %u resp2 = %u\n", __func__, resp.status, resp.resp0, resp.resp1, resp.resp2);
        return(ERROR);
    }
    udelay(20);

    memset(&rid, 0, sizeof(rid));

    rid.word = HFA384X_MTU;
    rid.len = sizeof(rid)/2 - 1;
    rid.rid = HFA384X_RID_CNFMAXDATALEN;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFMAXDATALEN, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set max data len address.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    rid.word = ai->channel;
    rid.len = sizeof(rid)/2 - 1;
    rid.rid = HFA384X_RID_CNFOWNCHANNEL;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNCHANNEL, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set channel.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    rid.word = 1;
    rid.len = sizeof(rid)/2 - 1;
    rid.rid = HFA384X_RID_CREATEIBSS;
    if(hfa384x_write_rid(ai, HFA384X_RID_CREATEIBSS, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set create ibss mode.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    rid.word = HFA384X_DISABLE_RXCRYPT|HFA384X_WEPFLAGS_HOSTENCRYPT|HFA384X_WEPFLAGS_HOSTDECRYPT;
    rid.len = sizeof(rid)/2 - 1;
    rid.rid = HFA384X_RID_CNFWEPFLAGS;
    /* turn off encryption */
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFWEPFLAGS, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set wep flags.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    udelay(20);
    rid.word = 100;
    rid.rid = HFA384X_RID_CNFBEACONINT;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFBEACONINT, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set beacon interval.\n", __func__);
        return(ERROR);
    }

    rid.word = 0;
    rid.len = sizeof(rid)/2 - 1;
    rid.rid = HFA384X_RID_CNFALTRETRYCOUNT;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFALTRETRYCOUNT, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set retry count.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    rid.word = ai->port_mode;
    rid.rid = HFA384X_RID_CNFPORTTYPE;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFPORTTYPE, &rid, sizeof(rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to change port type.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    memcpy(mac_rid.mac, ai->mac, sizeof(ai->mac));
    mac_rid.len = sizeof(mac_rid)/2 - 1;
    mac_rid.rid = HFA384X_RID_CNFOWNMACADDR;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNMACADDR, &mac_rid, sizeof(mac_rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set mac address.\n", __func__);
        return(ERROR);
    }
    udelay(20);

    memset(&essid_rid, 0, sizeof(essid_rid));
    memcpy(essid_rid.essid + 2, ai->essid + 1, ai->essid[0]);
    essid_rid.essid[0] = ai->essid[0];
    essid_rid.len = sizeof(essid_rid)/2 - 1;
    essid_rid.rid = HFA384X_RID_CNFOWNSSID;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFOWNSSID, &essid_rid, sizeof(essid_rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set essid.\n", __func__);
        return(ERROR);
    }
    essid_rid.rid = HFA384X_RID_CNFDESIREDSSID;
    if(hfa384x_write_rid(ai, HFA384X_RID_CNFDESIREDSSID, &essid_rid, sizeof(essid_rid), HFA384X_BAP1) != SUCCESS) {
        local_irq_enable();
        printk(KERN_ERR "airjack_cs: %s: failed to set essid.\n", __func__);
        return(ERROR);
    }
    udelay(20);
    
    local_irq_enable();
    DEBUG_PRINT("airjack_cs: %s: leaving.\n", __func__);
    return(SUCCESS);
}


/*
 * configures the hfa384x, hooks the device into the kernel...
 *
 ****
 * Call Context:
 *   User Context
 *
 */
static void airjack_config (dev_link_t *link)
{
    aj_info_t				*ai = (aj_info_t *)link->priv;
    client_handle_t			handle = link->handle;
    tuple_t					tuple;
    cisparse_t				parse;
	cistpl_cftable_entry_t	*cfg, dflt = { 0 };
    cisinfo_t				cis_info;
    config_info_t			conf;
    int						ret;
    unsigned char			buf[64];	/* XXX - magic number brought to you by David Hines */


    DEBUG_PRINT("airjack_cs: airjack_config: starting.\n");
    memset(&cis_info, 0, sizeof(cisinfo_t));

    /* make sure there is a valid CIS on this card */
    if((ret = CardServices(ValidateCIS, handle, &cis_info)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, ValidateCIS, ret);
        airjack_release((unsigned long int)link);
        return;
    }

    if(cis_info.Chains == 0) {
        printk(KERN_ERR "airjack_cs: airjack_config: No valid CIS found on card!\n");
        airjack_release((unsigned long int)link);
        return;
    }

    memset(&tuple, 0, sizeof(tuple_t));

    /* read the cards config tuple */
    tuple.DesiredTuple = CISTPL_CONFIG;
    tuple.Attributes = 0;
    tuple.TupleData = buf;
    tuple.TupleDataMax = sizeof(buf);
    tuple.TupleOffset = 0;
    if((ret = CardServices(GetFirstTuple, handle, &tuple)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, GetFirstTuple, ret);
        airjack_release((unsigned long int)link);
        return;
    }
    if((ret = CardServices(GetTupleData, handle, &tuple)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, GetTupleData, ret);
        airjack_release((unsigned long int)link);
        return;
    }
    if((ret = CardServices(ParseTuple, handle, &tuple, &parse)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, ParseTuple, ret);
        airjack_release((unsigned long int)link);
        return;
    }
    link->conf.ConfigBase = parse.config.base;
    link->conf.Present = parse.config.rmask[0];

    link->state |= DEV_CONFIG;

    if((ret = CardServices(GetConfigurationInfo, handle, &conf)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, GetConfigurationInfo, ret);
        airjack_release((unsigned long int)link);
        return;
    }

    tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
    if((ret = CardServices(GetFirstTuple, handle, &tuple)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, GetFirstTuple, ret);
        airjack_release((unsigned long int)link);
        return;
    }
    if((ret = CardServices(GetTupleData, handle, &tuple)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, GetTupleData, ret);
        airjack_release((unsigned long int)link);
        return;
    }
    if((ret = CardServices(ParseTuple, handle, &tuple, &parse)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, ParseTuple, ret);
        airjack_release((unsigned long int)link);
        return;
    }

    cfg = &(parse.cftable_entry);
    /* repeat as needed */
    while(cfg->index == 0) {
        if(cfg->flags & CISTPL_CFTABLE_DEFAULT) {
            dflt = *cfg;
        }
        if(link->io.NumPorts1) {
            CardServices(ReleaseIO, link->handle, &link->io);
        }
        if((ret = CardServices(GetNextTuple, handle, &tuple)) != CS_SUCCESS) {
            DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
            cs_error(handle, GetNextTuple, ret);
            airjack_release((unsigned long int)link);
            return;
        }
        if((ret = CardServices(GetTupleData, handle, &tuple)) != CS_SUCCESS) {
            DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
            cs_error(handle, GetTupleData, ret);
            airjack_release((unsigned long int)link);
            return;
        }
        if((ret = CardServices(ParseTuple, handle, &tuple, &parse)) != CS_SUCCESS) {
            DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
            cs_error(handle, ParseTuple, ret);
            airjack_release((unsigned long int)link);
            return;
        }
    }

    if(cfg->flags & CISTPL_CFTABLE_DEFAULT) {
        dflt = *cfg;
    }

    link->conf.ConfigIndex = cfg->index;

    /* Use power settings for Vcc and Vpp if present */
    /* Note that the CIS values need to be rescaled */
    if (cfg->vcc.present & (1<<CISTPL_POWER_VNOM)) {
        DEBUG_PRINT("vcc set from VNOM\n");
        link->conf.Vcc = cfg->vcc.param[CISTPL_POWER_VNOM]/10000;
    } else if (dflt.vcc.present & (1<<CISTPL_POWER_VNOM)) {
        DEBUG_PRINT("vcc set from VNOM\n");
        link->conf.Vcc = dflt.vcc.param[CISTPL_POWER_VNOM]/10000;
    } else if ((cfg->vcc.present & (1<<CISTPL_POWER_VMAX)) &&
           (cfg->vcc.present & (1<<CISTPL_POWER_VMIN)) ) {
        DEBUG_PRINT("vcc set from avg(VMIN,VMAX)\n");
        link->conf.Vcc = 
            ((cfg->vcc.param[CISTPL_POWER_VMAX] +
            cfg->vcc.param[CISTPL_POWER_VMIN]) / 2) / 10000;
    } else if ((dflt.vcc.present & (1<<CISTPL_POWER_VMAX)) &&
           (dflt.vcc.present & (1<<CISTPL_POWER_VMIN)) ) {
        DEBUG_PRINT("vcc set from avg(VMIN,VMAX\n");
        link->conf.Vcc = 
            ((dflt.vcc.param[CISTPL_POWER_VMAX] +
            dflt.vcc.param[CISTPL_POWER_VMIN]) / 2) / 10000;
    }

    /*
     * sense all the cards this works on are pcmcia
     * 5V cards im leaning towards Vcc = 50 here...
     */
    if((link->conf.Vcc <= 55) && (link->conf.Vcc >= 40))  {
        link->conf.Vcc = 50;
    } else {
        link->conf.Vcc = 33; 
    }

    link->conf.Attributes |= CONF_ENABLE_IRQ;

    /* configure the I/O window settings */
    link->io.NumPorts1 = link->io.NumPorts2 = 0;
    if((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
        cistpl_io_t	*io = (cfg->io.nwin) ? &cfg->io : &dflt.io;

        link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
        if (!(io->flags & CISTPL_IO_8BIT)) {
            link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
        }
        if(!(io->flags & CISTPL_IO_16BIT)) {
            DEBUG_PRINT("airjack_cs: airjack_config: shit, the card doesnt support 16 bit io\n");
            link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
        }
        link->io.IOAddrLines = io->flags & CISTPL_IO_LINES_MASK;
        link->io.BasePort1 = io->win[0].base;
        link->io.NumPorts1 = io->win[0].len;
        if(io->nwin > 1) {
            link->io.Attributes2 = link->io.Attributes1;
            link->io.BasePort2 = io->win[1].base;
            link->io.NumPorts2 = io->win[1].len;
        }
        /* This reserves IO space but doesn't actually enable it */
        if((ret = CardServices(RequestIO, handle, &link->io)) != CS_SUCCESS) {
            DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
            cs_error(handle, RequestIO, ret);
            airjack_release((unsigned long int)link);
            return;
        }
    }

    /* request an irq line */
    if((ret = CardServices(RequestIRQ, handle, &link->irq)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, RequestIRQ, ret);
        airjack_release((unsigned long int)link);
        return;
    }

    /* configure the PCMCIA socket */
    if((ret = CardServices(RequestConfiguration, handle, &link->conf)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: airjack_config: error.\n");
        cs_error(handle, RequestConfiguration, ret);
        airjack_release((unsigned long int)link);
        return;
    }

    if(setup_net_dev(ai, link->irq.AssignedIRQ, link->io.BasePort1) != SUCCESS) {
        airjack_release((unsigned long int)link);
        printk(KERN_ERR "airjack_cs: airjack_config: card configuration failed.\n");
        return;
    }

    if(request_irq(ai->net_dev->irq, airjack_interrupt, SA_SHIRQ, ai->net_dev->name, ai->net_dev) != 0) {
        printk(KERN_ERR "airjack_cs: airjack_config: irq request failed.\n");
        stop_airjack_card(ai->net_dev);
        airjack_release((unsigned long int)link);
        return;
    }

    ai->irq_set = 1;

    /* handles its own locking so as to prevent a deadlock */
    if(initial_config(ai) != SUCCESS) {
        printk(KERN_ERR "airjack_cs: airjack_config: card configuration failed.\n");
        stop_airjack_card(ai->net_dev);
        airjack_release((unsigned long int)link);
        return;
    }

    if(finish_bootstrap(ai) != SUCCESS) {
        printk(KERN_ERR "airjack_cs: airjack_config: card configuration failed.\n");
        stop_airjack_card(ai->net_dev);
        airjack_release((unsigned long int)link);
        return;
    }

    if(register_netdev(ai->net_dev) != 0) {
        printk(KERN_ERR "airjack_cs: airjack_config: failed to register net device.\n");
        stop_airjack_card(ai->net_dev);
        airjack_release((unsigned long int)link);
        return;
    }


    /* for this device we start in promisc mode from the begining */
    ai->net_dev->flags |= IFF_BROADCAST|IFF_MULTICAST|IFF_PROMISC|IFF_ALLMULTI|IFF_RUNNING;
    /* notify whoever that we put it in a new state */
    netdev_state_change(ai->net_dev);

    /* only registered here so no locking needed (i think) */
    ai->registered = 1;


    SET_MODULE_OWNER(ai->net_dev);

	strcpy(ai->node.dev_name, ai->net_dev->name);
	ai->node.major = ai->node.minor = 0;
	link->dev = &ai->node;


    link->state &= ~DEV_CONFIG_PENDING;
    DEBUG_PRINT("airjack_cs: airjack_config: Configuration Successful.\n");

    /* report what we've done */
    printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d",
           ai->node.dev_name, link->conf.ConfigIndex,
           link->conf.Vcc/10, link->conf.Vcc%10);
    if(link->conf.Vpp1) {
        printk(", Vpp %d.%d", link->conf.Vpp1/10, link->conf.Vpp1%10);
    }
    if(link->conf.Attributes & CONF_ENABLE_IRQ) {
        printk(", irq %d", link->irq.AssignedIRQ);
    }
    if(link->io.NumPorts1) {
        printk(", io 0x%04x-0x%04x", link->io.BasePort1,
               link->io.BasePort1+link->io.NumPorts1-1);
    }
    if(link->io.NumPorts2) {
        printk(" & 0x%04x-0x%04x", link->io.BasePort2,
               link->io.BasePort2+link->io.NumPorts2-1);
    }
    printk("\n");

}


/*
 * releases all resources allocated by card services for us,
 * schedules the device to be removed...
 */
static void airjack_release (unsigned long int arg)
{
    dev_link_t	*link = (dev_link_t *)arg;

    DEBUG_PRINT("airjack_cs: airjack_release: starting.\n");
    /* we'll wait till the device is closed before we actualy release it */
    if(link->open) {
        link->state |= DEV_STALE_CONFIG;
        DEBUG_PRINT("airjack_cs: airjack_release: defering release till the device isnt being used.\n");
        return;
    }

    link->dev = NULL;

    /* David Hines says I dont have to check if these succeed or not */
    if(link->win) {
        CardServices(ReleaseWindow, link->win);
    }
    CardServices(ReleaseConfiguration, link->handle);
    if(link->io.NumPorts1) {
        CardServices(ReleaseIO, link->handle, &link->io);
    }
    if(link->irq.AssignedIRQ) {
        CardServices(ReleaseIRQ, link->handle, &link->irq);
    }
    link->state &= ~DEV_CONFIG;
    DEBUG_PRINT("airjack_cs: airjack_release: airjack device should now be released.\n");
}


/*
 * this creates and instance of the driver, it sets up all 
 * data structures needed for the driver...
 *
 ****
 * Call Context:
 *  User Context
 *   
 */
static dev_link_t *airjack_attach (void)
{
    aj_info_t				*ai;
    dev_link_t				*link;
    client_reg_t			client_reg;
    int						ret, i;
    char					name[32];

    DEBUG_PRINT("airjack_cs: %s: starting.\n", __func__);
    
    flush_stale_links();

    /* allocate the airo info structure */
    if((ai = (aj_info_t *)kmalloc(sizeof(aj_info_t), GFP_KERNEL)) == NULL) {
        printk(KERN_ERR "airjack_cs: %s: Fatal Error: out of memory!...go eat worms\n", __func__);
        return(NULL);
    }
    memset(ai, 0, sizeof(aj_info_t));

    /* set defaults */
    ai->channel = DEFAULT_CHANNEL;
    ai->port_mode = 5;
    memset(ai->essid, 0, sizeof(ai->essid));
    memcpy(ai->essid + 1, "AirJack", 7);
    ai->essid[0] = 7;
    memcpy(ai->mac, mac_addr, sizeof(ai->mac));

    link = &ai->link;
    link->priv = ai;

    link->release.function = &airjack_release;
    link->release.data = (unsigned long int)link;

    /* interrupt setup */
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE; 
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    if(irq_list[0] == -1) {
        link->irq.IRQInfo2 = irq_mask;
    } else {
        for(i=0;i < 4;i++) {
            link->irq.IRQInfo2 |= 1 << irq_list[i];
        }
    }
    link->irq.Handler = NULL;

    /* a couple defaults */
    link->conf.Attributes = 0;
    link->conf.Vcc = HFA384X_VCC;
    link->conf.Vpp1 = HFA384X_VPP1;
    link->conf.Vpp2 = HFA384X_VPP2;
    link->conf.IntType = INT_MEMORY_AND_IO;

    /* link the device list entry into the device list */
    link->next = dev_list;
    dev_list = link;
    
    /* register with card services */
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask  = CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL;
    client_reg.EventMask |= CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET;
    client_reg.EventMask |= CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &airjack_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    if((ret = CardServices(RegisterClient, &link->handle, &client_reg)) != CS_SUCCESS) {
        DEBUG_PRINT("airjack_cs: %s: error.\n", __func__);
        cs_error(link->handle, RegisterClient, ret);
        airjack_detach(link);
        return(NULL);
    }

    sprintf(name, "%s_core", ai->net_dev->name);

    /* attach proc entries for core dumps */
    if((ai->proc_ent = create_proc_entry(name, S_IFREG | S_IRUSR, &proc_root)) == NULL) {
        return(NULL);
    }

    ai->proc_ent->proc_fops = &airjack_operations;
    ai->proc_ent->data = ai;
    ai->proc_ent->size = 1024*256;

    DEBUG_PRINT("airjack_cs: %s: leaving.\n", __func__);
    return(link);
}


/*
 * removes an instance of the driver, free's all allocated resources...
 */
static void airjack_detach (dev_link_t *link)
{
    dev_link_t	**linkp;


    DEBUG_PRINT("airjack_cs: airjack_detach: starting card detach.\n");
    /* find the address of the specified link structure... */
    for(linkp=&dev_list;(*linkp != NULL) && (*linkp != link);linkp = &(*linkp)->next);

    if(*linkp == NULL) {
        printk(KERN_ERR "airjack_cs: airjack_detach: Internal Error " __FILE__ " line %u: cant find the device link structure that im supposed to detach!.\n", __LINE__);
        return;
    }

    /* we dont want to detach it just yet if its in use */
    del_timer(&link->release);
    if(link->state & DEV_CONFIG) {
        airjack_release((unsigned long int)link);
        if(link->state & DEV_STALE_CONFIG) {
            link->state |= DEV_STALE_LINK;
            DEBUG_PRINT("airjack_cs: airjack_detach: detach deffered till card is not being used.\n");
            return;
        }
    }

    /* release our entry in /proc */
    remove_proc_entry(((aj_info_t *)(link->priv))->proc_ent->name, &proc_root);

    if(((aj_info_t *)(link->priv))->net_dev != NULL) {
        stop_airjack_card(((aj_info_t *)(link->priv))->net_dev);
    }
    ((aj_info_t *)(link->priv))->net_dev = NULL;

    /* de-register with Card Services */
    if(link->handle) {
        CardServices(DeregisterClient, link->handle);
    }

    /* unlink and free the device structure */
    *linkp = link->next;
    /* no need to kfree link after this because link is just link->priv->link */
    if(link->priv) {
        kfree(link->priv);
    }
    DEBUG_PRINT("airjack_cs: airjack_detach: finished card detach.\n");
}


/*
 * modules init entry point...
 *
 ****
 * Call Context:
 *  User Context, Module Load
 *   
 */
static int init_airjack (void)
{
    servinfo_t	serv;

    DEBUG_PRINT("airjack_cs: init_airjack: starting initialization\nairjack_cs was compiled on " __DATE__ " at " __TIME__ ".\n");
    if(mac_param != NULL) {
        if(string_to_mac(mac_param, mac_addr) != SUCCESS) {
            printk(KERN_WARNING "airjack_cs: airjack_config: invalid mac address specified, reverting to default.\n");
            memcpy(mac_addr, "\x00\xde\xad\xc0\xde\x00", sizeof(mac_addr));
        }
    }

    /* make sure its the same card services version */
    CardServices(GetCardServicesInfo, &serv);

    if(serv.Revision != CS_RELEASE_CODE) {
        printk(KERN_ERR "airjack_cs: airjack_config: incompatable Card Services version in use.\n");
        return(ERROR);
    }

    /* register with driver services */
    register_pccard_driver(&dev_info, &airjack_attach, &airjack_detach);

    printk(KERN_INFO "\nairjack_cs: Initialization complete.\n");
    return(SUCCESS);
}


/*
 * module cleanup function...
 *
 ****
 * Call Context:
 *  User Context, Module Unload
 *   
 */
static void cleanup_airjack (void)
{

    DEBUG_PRINT("airjack_cs: cleanup_airjack: starting cleanup\n");
    unregister_pccard_driver(&dev_info);
    while(dev_list != NULL) {
        del_timer(&dev_list->release);
        if(dev_list->state & DEV_CONFIG) {
            airjack_release((unsigned long int)dev_list);
        }
        airjack_detach(dev_list);
    }

    printk(KERN_INFO "\nairjack_cs: Cleanup complete.\n");
}

module_init(init_airjack);
module_exit(cleanup_airjack);
