#include <string>
#include <vector>

#include "main.h"

#include "SDL/SDL.h"



/*
	A little class that helps in keeping track of SDL joysticks.
*/
class Joystick
{
public:
	inline Joystick(void *_device, std::string _name,
		Uint32 _buttons, Uint32 _axes,
		Uint32 _hats, Uint32 _balls)
	{
		device = _device;
		name = _name;
		buttons = _buttons; axes = _axes; hats = _hats; balls = _balls;

		button = new Uint8[buttons]; buttonPrev = new Uint8[buttons];
		axis = new float[axes];
		hat = new Uint8[hats];
		ball_x = new float[balls]; ball_y = new float[balls];

		for (Uint32 i = 0; i < buttons; ++i) {button[i] = 0; buttonPrev[i] = 0;}
		for (Uint32 i = 0; i < axes; ++i) axis[i] = 0.0;
		for (Uint32 i = 0; i < hats; ++i) hat[i] = 0;
		for (Uint32 i = 0; i < balls; ++i) {ball_x[i] = 0.0; ball_y[i] = 0.0;}
	}
	inline ~Joystick()
		{delete[] button; delete[] axis; delete[] hat;
		delete[] ball_x; delete[] ball_y;}

	void *device;
	std::string name;
	Uint32 buttons, axes, hats, balls;
	Uint8 *button, *buttonPrev;
	float *axis;
	Uint8 *hat;
	float *ball_x, *ball_y;
};


static bool stickInit = false;
static std::vector<Joystick*> sticks;

#define INDEX(X) (((X) < 0.0) ? (0xFFFFFFFF) : (int((X)+.1)))

//----------------


GMReal joy_count()
{
	return double(sticks.size());
}

GMString joy_name(double id)
{
	Uint32 index = INDEX(id);
	if (index >= sticks.size()) return "(No such device)";
	return sticks[index]->name.c_str();
}

GMReal joy_axes(double id)
{
	Uint32 index = INDEX(id);
	if (index >= sticks.size()) return 0.0;
	return sticks[index]->axes;
}

GMReal joy_axis(double id, double axis)
{
	Uint32 index = INDEX(id), axid = INDEX(axis);
	if (index >= sticks.size()) return 0.0;
	if (axid >= sticks[index]->axes) return 0.0;
	return sticks[index]->axis[axid];
}

GMReal joy_buttons(double id)
{
	Uint32 index = INDEX(id);
	if (index >= sticks.size()) return 0.0;
	return sticks[index]->buttons;
}

GMReal joy_button(double id, double button)
{
	Uint32 index = INDEX(id), butid = INDEX(button);
	if (index >= sticks.size()) return 0.0;
	if (butid >= sticks[index]->buttons) return 0.0;
	return (sticks[index]->button[butid]) ? 1.0 : 0.0;
}

GMReal joy_pressed(double id, double button)
{
	Uint32 index = INDEX(id), butid = INDEX(button);
	if (index >= sticks.size()) return 0.0;
	if (butid >= sticks[index]->buttons) return 0.0;
	return (sticks[index]->button[butid] &&
			!sticks[index]->buttonPrev[butid]) ? 1.0 : 0.0;
}

GMReal joy_released(double id, double button)
{
	Uint32 index = INDEX(id), butid = INDEX(button);
	if (index >= sticks.size()) return 0.0;
	if (butid >= sticks[index]->buttons) return 0.0;
	return (!sticks[index]->button[butid] &&
			sticks[index]->buttonPrev[butid]) ? 1.0 : 0.0;
}

GMReal joy_hats(double id)
{
	Uint32 index = INDEX(id);
	if (index >= sticks.size()) return 0.0;
	return sticks[index]->hats;
}

GMReal joy_hat(double id, double hat)
{
	Uint32 index = INDEX(id), hatid = INDEX(hat);
	if (index >= sticks.size()) return -1.0;
	if (hatid >= sticks[index]->hats) return -1.0;
	switch(sticks[index]->hat[hatid])
	{
	case SDL_HAT_UP:
		return 0.0f;
		break;
	case SDL_HAT_RIGHT:
		return 90.0f;
		break;
	case SDL_HAT_DOWN:
		return 180.0f;
		break;
	case SDL_HAT_LEFT:
		return 270.0f;
		break;
	case SDL_HAT_RIGHTUP:
		return 45.0f;
		break;
	case SDL_HAT_RIGHTDOWN:
		return 135.0f;
		break;
	case SDL_HAT_LEFTUP:
		return 315.0f;
		break;
	case SDL_HAT_LEFTDOWN:
		return 225.0f;
		break;
	case SDL_HAT_CENTERED:
	default:
		return -1.0f;
	}
}

GMReal joy_hat_x(double id, double hat)
{
	Uint32 index = INDEX(id), hatid = INDEX(hat);
	if (index >= sticks.size()) return 0.0;
	if (hatid >= sticks[index]->hats) return 0.0;
	Uint32 pos = sticks[index]->hat[hatid];
	if (pos & SDL_HAT_LEFT) return -1.0;
	if (pos & SDL_HAT_RIGHT) return 1.0;
	return 0.0;
}

GMReal joy_hat_y(double id, double hat)
{
	Uint32 index = INDEX(id), hatid = INDEX(hat);
	if (index >= sticks.size()) return 0.0;
	if (hatid >= sticks[index]->hats) return 0.0;
	Uint32 pos = sticks[index]->hat[hatid];
	if (pos & SDL_HAT_UP) return -1.0;
	if (pos & SDL_HAT_DOWN) return 1.0;
	return 0.0;
}

GMReal joy_balls(double id)
{
	Uint32 index = INDEX(id);
	if (index >= sticks.size()) return 0.0;
	return sticks[index]->balls;
}

GMReal joy_ball_x(double id, double ball)
{
	Uint32 index = INDEX(id), ballid = INDEX(ball);
	if (index >= sticks.size()) return 0.0;
	if (ballid >= sticks[index]->balls) return 0.0;
	return sticks[index]->ball_x[ballid];
}

GMReal joy_ball_y(double id, double ball)
{
	Uint32 index = INDEX(id), ballid = INDEX(ball);
	if (index >= sticks.size()) return 0.0;
	if (ballid >= sticks[index]->balls) return 0.0;
	return sticks[index]->ball_y[ballid];
}



//----------------------

double joy_refresh()
{
	//And figure out what's currently connected
	Uint16 len = SDL_NumJoysticks();
	SDL_Joystick *device;
	for(Uint16 i = 0; i < len; i++)
	{
		device = SDL_JoystickOpen(i);
		if (SDL_JoystickOpened(i))
		{
			Joystick *stick = new Joystick((void*)device, SDL_JoystickName(i),
					SDL_JoystickNumButtons(device), SDL_JoystickNumAxes(device),
					SDL_JoystickNumHats(device), SDL_JoystickNumBalls(device));
			sticks.push_back(stick);
		}
	}

	return sticks.size();
}

GMReal joy_init()
{
	if (stickInit) return 0.0;

	stickInit = true;

	//Start up the SDL subsystem
	SDL_Init(SDL_INIT_JOYSTICK);

	//Detect joysticks
	joy_clear();
	joy_refresh();

	//We're probably okay
	return 1.0;
}

GMReal joy_close()
{
	if (!stickInit) return 0.0;

	//Clean up
	joy_clear();

	//Close the SDL subsystem
	SDL_Quit();

	stickInit = false;

	//We're probably ok
	return 1.0;
}

void joy_clear()
{
	//Delete all the stick objects
	Uint16 len = sticks.size();
	for(Uint16 i = 0; i < len; i++)
	{
		SDL_JoystickClose((SDL_Joystick*)sticks[i]->device);
		delete sticks[i];
	}
	sticks.clear();
}

GMReal joy_find()
{
	//Reset
	if (stickInit)
	{
		joy_close();
		joy_init();
	}
	else return -1.0;

	joy_clear();
	return joy_refresh();
}

GMReal joy_update()
{
	//Track lost sticks
	bool lost = false;

	//Joystick input update
	SDL_JoystickUpdate();
	{
		Joystick *joy;
		SDL_Joystick *device;
		int ballx, bally;
		Uint32 limit;
		for (std::vector<Joystick*>::iterator iter = sticks.begin();
			iter != sticks.end(); iter++)
		{
			joy = *iter;
			device = (SDL_Joystick*)joy->device;
			//If joystick is disconnected then deactivate and enter dummy data
			if (device==NULL || !SDL_JoystickOpened(SDL_JoystickIndex(device)))
			{
				lost = true;

				joy->device = NULL;
				limit = joy->buttons;
				for (Uint32 i = 0; i < limit; i++)
					{joy->buttonPrev[i] = joy->button[i]; joy->button[i] = 0;}
				limit = joy->axes;
				for (Uint32 i = 0; i < limit; i++)
					joy->axis[i] = 0.0f;
				limit = joy->hats;
				for (Uint32 i = 0; i < limit; i++)
					joy->hat[i] = SDL_HAT_CENTERED;
				limit = joy->balls;
				for (Uint32 i = 0; i < limit; i++)
					joy->ball_x[i] = 0.0f, joy->ball_y[i] = 0.0f;
				continue;
			}
			limit = joy->buttons;
			for (Uint32 i = 0; i < limit; i++)
				{joy->buttonPrev[i] = joy->button[i];
				joy->button[i] = SDL_JoystickGetButton(device, i);}
			limit = joy->axes;
			for (Uint32 i = 0; i < limit; i++)
				joy->axis[i] =
					float(SDL_JoystickGetAxis(device, i)) / 32768.0f;
			limit = joy->hats;
			for (Uint32 i = 0; i < limit; i++)
				joy->hat[i] = SDL_JoystickGetHat(device, i);
			limit = joy->balls;
			for (Uint32 i = 0; i < limit; i++)
			{
				SDL_JoystickGetBall(device, i, &ballx, &bally);
				joy->ball_x[i] = float(ballx);
				joy->ball_y[i] = float(bally);
			}
		}
	}

	return lost;
}

//----------------------


GMString joy_hi()
{
	return "Welcome to joyDLL for Game Maker!";
}
