#include "Default.h"
#include <fat.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h> //for qsort

#include "EmuSystem.h"
#include "NeoSystem.h"
#include "NeoSystemAsm.h"
#include "pd4990a.h"
#include "NeoCpu.h"
#include "NeoVideo.h"
#include "NeoMemory.h"
#include "NeoIPC.h"
#include "NeoAudioStream.h"
#include "NeoProfiler.h"
#include "LinearHeap.h"
#include "guiConsole.h"
#include "guiBase.h"

#define NEO_HEADER_SIZE 512

STATIC_ASSERT(sizeof(TNeoContext) == NEO_CONTEXTEND);
//verifty that the variable section of TNeoContext lines up (without the mem tables)
STATIC_ASSERT(OFFSET(TNeoContext, varEnd) == NEO_VAREND);
//verifty that varEnd is less than 0x400
STATIC_ASSERT(OFFSET(TNeoContext, cpuRead8Table) == 0x400);

TNeoContext g_neoContext DTCM_BSS;
//TNeoContext* g_neo DTCM_BSS;

TNeoRomHeader g_header;
static int g_rom = -1;
static u32 g_romSize = 0;
static TLinearHeap g_vramHHeap;

static u32 g_romCount = 0;
static char** g_romNames = 0;

#ifndef NEO_SHIPPING
static const char* const g_neoRomRegionNames[] = {
	"MainProgram",
	"Bios",
	"AudioProgram",
	"FixedData",
	"AudioData1",
	"AudioData2",
	"AudioData3",
	"AudioData4",
	"SpriteData",
};
#endif

void neoSystemIPCSync()
{
	/*u32 command = IPC_GetSync();
	switch(command) {
	case NEOARM9_READAUDIO:
		neoIPCRecvCommand();
		NEOIPC->arm9Return = neoAudioStream(NEOIPC->arm9Args[0], &NEOIPC->arm9Args[1]);
		neoIPCAckCommand();
		break;
	}*/
}

void neoSystemReadRom(u8* pDst, u32 offset, u32 size)
{
	profilerPush(NEOPROFILE_ROMREAD);
	systemSlot2Lock();
	//lock
	irqDisable(IRQ_VBLANK | IRQ_VCOUNT);
	//read
	lseek(g_rom, offset, SEEK_SET);
	read(g_rom, pDst, size);
	//unlock
	irqEnable(IRQ_VBLANK | IRQ_VCOUNT);
	systemSlot2Unlock();
	profilerPop();
}

void neoSystemLoadSprite(u8* pDst, u32 index)
{
	const u32 romOffset =
		g_header.romEntry[NEOROM_SPRITEDATA].offset +
		index * SPRITE_CACHE_ENTRY_SIZE;
	neoSystemReadRom(pDst, romOffset, SPRITE_CACHE_ENTRY_SIZE);
}

void neoSystemLoadSprite2(u8* pDst, u32 index)
{
	const u32 romOffset =
		g_header.romEntry[NEOROM_SPRITEDATA].offset +
		index * SPRITE_CACHE2_ENTRY_SIZE;
	neoSystemReadRom(pDst, romOffset, SPRITE_CACHE2_ENTRY_SIZE);
}

void neoSystemLoadTile(u8* pDst, u32 index)
{
	const u32 romOffset =
		g_header.romEntry[NEOROM_FIXEDDATA].offset +
		index * TILE_CACHE_ENTRY_SIZE;
	neoSystemReadRom(pDst, romOffset, TILE_CACHE_ENTRY_SIZE);
}

void neoSystemLoadRegionEx(TNeoRomRegion region, void* pDst, u32 offset, u32 maxSize)
{
	const TNeoRomEntry* pRegion = &g_header.romEntry[(u32)region];
	const u32 fileOffset = pRegion->offset + offset;
	u32 size = maxSize;
	if(offset >= pRegion->size) return;
	if(offset + size > pRegion->size) size = pRegion->size - offset;

	if(pRegion->offset == 0xffffffff || pRegion->size == 0xffffffff) {
		systemWriteLine("Undefined region");
		return;
	}
	neoSystemReadRom(pDst, fileOffset, size);
}

void neoSystemReadRegion(TNeoRomRegion region, void* pDst, u32 offset, u32 size)
{
	const TNeoRomEntry* pRegion = &g_header.romEntry[(u32)region];
	const u32 fileOffset = pRegion->offset + offset;
	neoSystemReadRom(pDst, fileOffset, size);
}

u32 neoSystemRegionSize(TNeoRomRegion region)
{
	const TNeoRomEntry* pRegion = &g_header.romEntry[(u32)region];
	return pRegion->size;
}

void neoSystemLoadRegion(TNeoRomRegion region, void* pDst, u32 maxSize)
{
	neoSystemLoadRegionEx(region, pDst, 0, maxSize);
}

void neoInstallProtection();

void* neoSystemVramHAlloc(u32 size)
{
	return linearHeapAlloc(&g_vramHHeap, size);
}

/*typedef void (*TRomFileIterator)(const char* szFileName, void* arg);

static bool romFileIterate(TRomFileIterator iterator, void* arg)
{
	DIR_ITER* dir;
	struct stat st;
	char szFilename[256];

	dir = diropen("/");
	if(!dir) {
		return false;
	}

	while(dirnext(dir, szFilename, &st) == 0) {
		if(st.st_mode & S_IFDIR) {
			continue;
		}
		const char* szExt = strchr(szFilename, '.');
		if(strcmpi(".NEO", szExt)) {
			continue;
		}
		iterator(szFilename, arg);
	}

	dirclose(dir);
	return true;
}

static void romCountIterator(const char* szFileName, void* arg)
{
	s32* pCount = (s32*)arg;
	*pCount = *pCount + 1;
}

static void romAddMenuIterator(const char* szFileName, void* arg)
{
	TGuiMenu* pMenu = (TGuiMenu*)arg;
	guiMenuAddItem(pMenu, szFileName);
}*/

int stringCompare(const void* a, const void* b)
{
	return strcmp(*(const char**)a, *(const char**)b);
}

bool neoSystemInit()
{
	g_neo = &g_neoContext;
	memset(g_neo, 0, sizeof(TNeoContext));

	irqSet(IRQ_IPC_SYNC, neoSystemIPCSync);

	linearHeapInit(&g_vramHHeap, (void*)0x6898000, 32*KB);

	systemWriteLine("sizeof(TNeoContext): %d", sizeof(TNeoContext));
	systemWriteLine(" -> varEnd: %d", OFFSET(TNeoContext, varEnd));

	g_romCount = 0;
	g_romNames = 0;

	DIR_ITER* dir;
	struct stat st;
	char szFilename[256];

	dir = diropen("/");
	if(!dir) {
		return false;
	}

	while(dirnext(dir, szFilename, &st) == 0) {
		if(st.st_mode & S_IFDIR) {
			continue;
		}
		const char* szExt = strchr(szFilename, '.');
		if(strcmpi(".NEO", szExt)) {
			continue;
		}
		
		g_romCount++;
		g_romNames = (char**)systemRealloc(g_romNames, g_romCount * sizeof(char*));
		ASSERT(g_romNames);
		g_romNames[g_romCount - 1] = strdup(szFilename);
		ASSERT(g_romNames[g_romCount - 1]);
	}
	qsort(g_romNames, g_romCount, sizeof(const char*), stringCompare);

	dirclose(dir);

	g_neo->cpuClockDivide = 2;

	return true;
}

u32 neoSystemGetRomCount()
{
	return g_romCount;
}

const char* neoSystemGetRomName(u32 i)
{
	ASSERT(i < g_romCount);
	return g_romNames[i];
}

bool neoSystemOpen(const char* szFileName)
{
	TNeoRomHeader header;
	struct stat romStat;
	int rom;
	u32 bit;

	ASSERT(g_neo->cpuClockDivide == 2 || g_neo->cpuClockDivide == 3);

	guiConsoleLogf("Loading %s...", szFileName);
	guiConsoleDump();

	rom = open(szFileName, O_RDONLY);

	ASSERTMSG(g_neo->cpuClockDivide == 2 || g_neo->cpuClockDivide == 3, "%d", g_neo->cpuClockDivide);

	if(rom < 0) {
		guiConsoleLogf(" -> Failed!");
		ASSERT(0);
		return false;
	}
	read(rom, &header, sizeof(TNeoRomHeader));

	if(header.magic != NEO_ROM_MAGIC ||
		header.version != NEO_ROM_VERSION ||
		header.sectionCount != NEOROM_COUNT) {

		close(rom);
		guiConsoleLogf(" -> invalid rom");
		guiConsoleLogf(" -> magic %08X / %08X", header.magic, NEO_ROM_MAGIC);
		guiConsoleLogf(" -> version %d / %d", header.version, NEO_ROM_VERSION);
		guiConsoleLogf(" -> section %d / %d", header.sectionCount, NEOROM_COUNT);
		ASSERT(0);
		return false;
	}

	neoSystemClose();

	fstat(rom, &romStat);
	g_romSize = romStat.st_size - NEO_HEADER_SIZE;
	g_header = header;
	g_rom = rom;

	linearHeapReset(&g_vramHHeap);
	linearHeapClear(&g_vramHHeap);

	g_neo->scanline = 0;
	g_neo->frameCount = 0;
	g_neo->irqPending = 0;
	g_neo->paletteBank = 0;
	g_neo->fixedBank = 0;
	g_neo->sramProtectCount = 0;
	g_neo->debug = true;

	g_neo->irqVectorLatch = false;
	g_neo->screenDarkLatch = false;
	g_neo->fixedRomLatch = false;
	g_neo->sramProtectLatch = false;
	g_neo->paletteRamLatch = false;
	g_neo->smaRand = 0x2345;
	NEOIPC->audioCommandPending = 0;
	NEOIPC->audioResult = 0;
	
	//sram hack gets around watchdog protection check...values taken from GnGeo
	//added samsho5 variations
	g_neo->sramProtection = g_header.sramProtection;

	guiConsoleLogf("Loaded Game: %s", g_header.name);
	
	g_neo->spriteCount = g_header.romEntry[NEOROM_SPRITEDATA].size / SPRITE_SIZE;
	g_neo->spriteMask = 0xffffffff;
	for(bit = 0x80000000; bit != 0; bit >>= 1) {
		if((g_neo->spriteCount - 1) & bit) break;
		g_neo->spriteMask >>= 1;
	}

	g_neo->romBankCount = 0;
	const s32 bankSize = (s32)g_header.romEntry[NEOROM_MAINPROGRAM].size - 1*MB;
	if(bankSize > 0) {
		g_neo->romBankCount = bankSize / (1*MB);
		if(g_neo->romBankCount * 1*MB != bankSize) {
			g_neo->romBankCount++;
			guiConsoleLogf(" -> partial rom bank");
		}
	}

	guiConsoleLogf(" -> sprites: %d", g_neo->spriteCount);
	guiConsoleLogf(" -> mask: %08X", g_neo->spriteMask);
	guiConsoleLogf(" -> rom bank(s): %d", g_neo->romBankCount);
	if(g_neo->sramProtection == -1) {
		guiConsoleLogf(" -> sramProtection off");
	} else {
		guiConsoleLogf(" -> sramProtection: %08X", g_neo->sramProtection);
	}
	guiConsoleDump();

	systemRamReset();
	systemSlot2Reset();

	cpuInit();
	neoMemoryInit();
	neoIOInit();
	pd4990a_init();
	neoVideoInit();
	neoInstallProtection();
	//neoSystemIrq(INTR_COLDBOOT);
	if(NEOIPC->globalAudioEnabled) {
		neoAudioStreamInit();
		neoIPCSendCommand(NEOARM7_RESET);
	} else {
		//this is the value that needs to be set to bypass Z80 check
		NEOIPC->audioResult = 1;
	}
	cpuReset();

	ASSERT(g_neo->cpuClockDivide == 2 || g_neo->cpuClockDivide == 3);
	
	return true;
}

void neoSystemClose()
{
	if(g_rom >= 0) {
		close(g_rom);
		g_rom = -1;
	}
}

static inline void neoSystemUpdateIrq()
{
	if(g_neo->irqPending & INTR_COLDBOOT) {
		cpuInterrupt(4);
	} else if(g_neo->irqPending & INTR_DISPLAYPOS) {
		cpuInterrupt(2);
	} else if(g_neo->irqPending & INTR_VBLANK) {
		cpuInterrupt(1);
	} else {
		cpuInterrupt(0);
	}
}

void neoSystemIrqAk(u16 data)
{
	if(data & 0x01) g_neo->irqPending &= ~INTR_COLDBOOT;
	if(data & 0x02) g_neo->irqPending &= ~INTR_DISPLAYPOS;
	if(data & 0x04) g_neo->irqPending &= ~INTR_VBLANK;
	neoSystemUpdateIrq();
}

void neoSystemIrq(u32 irq)
{
	g_neo->irqPending |= irq;
	neoSystemUpdateIrq();
}

static void neoSystemRun(s32 pixelCycles)
{
	g_neo->displayCounter -= pixelCycles;

	if(g_neo->displayCounter <= 0) {
		
		if((g_neo->displayControl & (1 << 7)) && g_neo->displayCounterLoad > 0) {
			//reload display counter
			while(g_neo->displayCounter < 0) {
				g_neo->displayCounter += g_neo->displayCounterLoad;
			}
		} else {
			//doesn't reload
			g_neo->displayCounter = 0;
		}
		if(g_neo->displayControl & (1 << 4)) {
			//generate interrupt if needed
			neoSystemIrq(2);
		}
	}
}

void neoSystemReset()
{
	systemWriteLine("System reset");

	cpuReset();

	if(g_neo->irqVectorLatch) {
		g_neo->irqVectorLatch = false;
		neoSystemLoadRegion(NEOROM_BIOS, g_neo->pRom0, 128);
	}
	
	g_neo->screenDarkLatch = false;
	g_neo->sramProtectLatch = false;
	g_neo->paletteRamLatch = false;
	g_neo->paletteBank = 0;
	g_neo->watchdogCounter = 0;
}

void neoSystemSetEnabled(bool enable)
{
	if(enable != g_neo->active) {
		g_neo->active = enable;
		if(NEOIPC->globalAudioEnabled) {
			//only turn on arm7 if audio is on
			if(enable) neoIPCSendCommand(NEOARM7_RESUME);
			else neoIPCSendCommand(NEOARM7_PAUSE);
		}
	}
}

s32 neoGetScanline(s32 cycles)
{
	/*static s32 scanline = 0;
	scanline++;
	if(scanline >= SCANLINES_PER_FRAME) {
		scanline = 0;
	}
	return scanline;*/
	return div32(cycles, CPU_CLOCKS_PER_SCANLINE);
}

static void neoSystemDoFrame()
{
	u32 keys;
	u8 input;

	ASSERT(g_neo->cpuClockDivide == 2 || g_neo->cpuClockDivide == 3);

	profilerPush(NEOPROFILE_CPU);
	/*if(g_neo->displayControl & (1 << 4)) {
		s32 cycles = CPU_CLOCKS_PER_FRAME;
		while(cycles > 0) {
			s32 cyclesToRun = g_neo->displayCounter / PIXELS_PER_CLOCK;
			if(cyclesToRun < CPU_CLOCKS_PER_SCANLINE) cycles = CPU_CLOCKS_PER_SCANLINE;
			if(cyclesToRun > cycles) cyclesToRun = cycles;

			systemWriteLine("blank intr %d/%d", cyclesToRun, cycles);
			cycles -= cpuExecute(cyclesToRun);
			neoSystemRun(cyclesToRun * PIXELS_PER_CLOCK);
		}
	} else {*/
		cpuExecute(CPU_CLOCKS_PER_FRAME);
		neoSystemRun(PIXELS_PER_FRAME);
	//}
	profilerPop();

	g_neo->scanline++;
	if(g_neo->scanline >= SCANLINES_PER_FRAME) {
		g_neo->scanline = 0;
	}
	g_neo->watchdogCounter++;
	/*if(g_neo->watchdogCounter > 10) {
		systemWriteLine("WATCHDOG!!!!");
		neoSystemReset();
		return;
	}*/

	neoAudioStreamProcess();
	neoVideoDrawFrame();
	neoAudioStreamProcess();

	g_neo->frameCount++;
	if(g_neo->displayControl & (1 << 6)) {
		g_neo->displayCounter = g_neo->displayCounterLoad;
	}
	g_neo->frameCounter--;
	if(g_neo->frameCounter == 0xff) {
		g_neo->frameCounter = (u8)(g_neo->displayControl >> 8);
		g_neo->autoAnimationCounter =
			(g_neo->autoAnimationCounter + 1) & 0x07;
	}

	pd4990a_addretrace();
	neoSystemIrq(1); //vblank

	keys = keysHeld();
	
	input = 0xff;
	if(keys & KEY_UP) input &= ~(1 << 0);
	if(keys & KEY_DOWN) input &= ~(1 << 1);
	if(keys & KEY_LEFT) input &= ~(1 << 2);
	if(keys & KEY_RIGHT) input &= ~(1 << 3);
	if(keys & KEY_A) input &= ~(1 << 4);
	if(keys & KEY_B) input &= ~(1 << 5);
	if(keys & KEY_X) input &= ~(1 << 6);
	if(keys & KEY_Y) input &= ~(1 << 7);
	g_neo->ctrl1Reg = 0x00ff | ((u16)input << 8);

	g_neo->ctrl2Reg = 0xffff;

	input = 0x05;
	if(keys & KEY_START) input &= ~(1 << 0);
	g_neo->ctrl3Reg = 0x8A00 | 0x7000 | ((u16)input << 8);

	input = 0x03;
	if(keys & KEY_SELECT) input &= ~(1 << 0);
	//set audio command to 01 to hack out sound
	g_neo->coinReg = 0x0004 | input |
		(read_4990_testbit() << 6) | (read_4990_databit() << 7);// | 
		//((u16)NEOIPC->audioResult << 8);
	/*if(!NEOIPC->globalAudioEnabled) {
		//hack audio ready bit
		g_neo->coinReg |= (1 << 8);
	}*/

	g_neo->ctrl4Reg = 0xff80;
}

void neoSystemExecute()
{
	profilerPush(NEOPROFILE_ROOT);
	while(1) {
		if(g_neo->active) {
			neoSystemDoFrame();
		} else {
			neoAudioStreamProcess();
			swiWaitForVBlank();
			guiSystemRender();
		}
		guiSystemProcess();
	}
	profilerPop();
}

	
		
