//#pragma once

#ifdef _MSC_VER
  #if (_MSC_VER >= 1200)
  #define M_INLINE __forceinline
  #else
  #define M_INLINE __inline
  #endif
#else
  #ifdef __cplusplus
  #define M_INLINE inline
  #else
  #define M_INLINE
  #endif
#endif

// ------------------------------------------------------------------------ //

template<class Type>
M_INLINE Type sqr(const Type& value) {return value*value;}

// ------------------------------------------------------------------------ //
// Fast Math
// ------------------------------------------------------------------------ //

/*
* Haven't seen this elsewhere, probably because it is too obvious?
* Anyway, these functions are intended for 32-bit floating point numbers
* only and should work a bit faster than the regular ones.
*/
//M_INLINE
//float fabsf(const float f)
//{
//	int i=((*(int*)&f)&0x7fffffff);
//	return (*(float*)&i);
//}

//M_INLINE
//float fast_neg(const float f)
//{
//	int i=((*(int*)&f)^0x80000000);
//	return (*(float*)&i);
//}

//M_INLINE
//int fast_sgn(const float f)
//{
//	return 1+(((*(int*)&f)>>31)<<1);
//}

/*
* Linear approx. between 2 integer values of val. Uses 32-bit integers.
* Not very efficient but faster than exp()
*/
//double fast_exp2( const double val )
//{
//    int e;
//    double ret;
//
//    if (val >= 0) {
//        e = int (val);
//        ret = val - (e - 1);
//
//#if BYTE_ORDER == BIG_ENDIAN
//        ((*((int *) &ret)) &= ~(2047 << 20)) += (e + 1023) << 20;
//#else
//        ((*(1 + (int *) &ret)) &= ~(2047 << 20)) += (e + 1023) << 20;
//#endif
//    } else {
//        e = int (val + 1023);
//        ret = val - (e - 1024);
//
//#if BYTE_ORDER == BIG_ENDIAN
//        ((*((int *) &ret)) &= ~(2047 << 20)) += e << 20;
//#else
//        ((*(1 + (int *) &ret)) &= ~(2047 << 20)) += e << 20;
//#endif
//    }
//
//    return ret;
//}
//
//M_INLINE
//float _fast_log2(const float val)
//{
//	float result, tmp;
//	float mp = 0.346607f;
//
//	result = (float)(*(int*)&val);
//	result *= 1.0/(1<<23);
//	result = result - 127;
//
//	tmp = result - floorf(result);
//	tmp = (tmp - tmp*tmp) * mp;
//	return tmp + result;
//}
//
//M_INLINE
//float _fast_pow2(const float val)
//{
//	float result;
//
//	float mp = 0.33971f;
//	float tmp = val - floorf(val);
//	tmp = (tmp - tmp*tmp) * mp;
//
//	result = val + 127 - tmp; 
//	result *= (1<<23);
//	*(int*)&result = (int)result;
//	return result;
//}

/*
* fastpow(f,n) gives a rather *rough* estimate of a float number f to the
* power of an integer number n (y=f^n). It is fast but result can be quite a
* bit off, since we directly mess with the floating point exponent.
* 
* Use it only for getting rough estimates of the values and where precision 
* is not that important.
*/
M_INLINE
float fast_pow(const float f, const int n)
{
	long *lp,l;
	lp=(long*)(&f);
	l=*lp;l-=0x3F800000l;l<<=(n-1);l+=0x3F800000l;
	*lp=l;
	return f;
}

M_INLINE
float fast_root(const float f, const int n)
{
	long *lp,l;
	lp=(long*)(&f);
	l=*lp;l-=0x3F800000l;l>>=(n-1);l+=0x3F800000l;
	*lp=l;
	return f;
}

M_INLINE
float fast_sqrt(const float f)
{
	long *lp, l;
	lp = (long*)(&f);
	l = *lp;l -= 0x3F800000l; l >>= 1;l += 0x3F800000l;
	*lp = l;
	return f;
}


/*
* Code for approximation of cos, sin, tan and inv sin, etc.
* Surprisingly accurate and very usable.
*
* Domain:
* Sin/Cos    [0, pi/2]
* Tan        [0,pi/4]
* InvSin/Cos [0, 1]
* InvTan     [-1, 1]
*/

M_INLINE
float fast_sin(const float val)
{
	float fASqr = val*val;
	float fResult = -2.39e-08f;
	fResult *= fASqr;
	fResult += 2.7526e-06f;
	fResult *= fASqr;
	fResult -= 1.98409e-04f;
	fResult *= fASqr;
	fResult += 8.3333315e-03f;
	fResult *= fASqr;
	fResult -= 1.666666664e-01f;
	fResult *= fASqr;
	fResult += 1.f;
	fResult *= val;

	return fResult;
}

M_INLINE
float fast_cos(const float val)
{
	float fASqr = val*val;
	float fResult = -2.605e-07f;
	fResult *= fASqr;
	fResult += 2.47609e-05f;
	fResult *= fASqr;
	fResult -= 1.3888397e-03f;
	fResult *= fASqr;
	fResult += 4.16666418e-02f;
	fResult *= fASqr;
	fResult -= 4.999999963e-01f;
	fResult *= fASqr;
	fResult += 1.f;

	return fResult;  
}

M_INLINE
float fast_tan(const float val)
{
	float fASqr = val*val;
	float fResult = 9.5168091e-03f;
	fResult *= fASqr;
	fResult += 2.900525e-03f;
	fResult *= fASqr;
	fResult += 2.45650893e-02f;
	fResult *= fASqr;
	fResult += 5.33740603e-02f;
	fResult *= fASqr;
	fResult += 1.333923995e-01f;
	fResult *= fASqr;
	fResult += 3.333314036e-01f;
	fResult *= fASqr;
	fResult += 1.f;
	fResult *= val;

	return fResult;

}

M_INLINE
float fast_atan2(float x, float y)
{
	float d = x*x + y*y;
	if (d == 0) return 0;
	float val = x / sqrtf(d);
	if (x >= 0) {
		float fAng = (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val);
		return (y >= 0) ? M_PI_2f - fAng : M_PI_2f + fAng;
	} else {
		val = -val;
		float fAng = (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val);
		return (y >= 0) ? fAng - M_PI_2f : -M_PI_2f - fAng;
	}
}

M_INLINE
float fast_asin(float val)
{
	if (val >= 0) {
		return M_PI_2f - (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val);
	} else {
		val = -val;
		return (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val) - M_PI_2f;
	}
}

M_INLINE
float fast_acos(float val)
{
	if (val >= 0) {
		return (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val);
	} else {
		val = -val;
		return M_PIf - (((-0.0187293f*val + 0.0742610f)*val - 0.2121144f)*val + 1.5707288f)*sqrtf(1.f-val);
	}
}

M_INLINE
float fast_atan(float val)
{
	float fVSqr = val*val;
	float fResult = 0.0028662257f;
	fResult *= fVSqr;
	fResult -= 0.0161657367f;
	fResult *= fVSqr;
	fResult += 0.0429096138f;
	fResult *= fVSqr;
	fResult -= 0.0752896400f;
	fResult *= fVSqr;
	fResult += 0.1065626393f;
	fResult *= fVSqr;
	fResult -= 0.1420889944f;
	fResult *= fVSqr;
	fResult += 0.1999355085f;
	fResult *= fVSqr;
	fResult -= 0.3333314528f;
	fResult *= fVSqr;
	fResult += 1.f;
	fResult *= val;

	return fResult;
}

//M_INLINE
//float fast_log2 (float val)
//{
//	int * const    exp_ptr = reinterpret_cast <int *> (&val);
//	int            x = *exp_ptr;
//	const int      log_2 = ((x >> 23) & 255) - 128;
//	x &= ~(255 << 23);
//	x += 127 << 23;
//	*exp_ptr = x;
//
//	val = ((-1.f/3) * val + 2) * val - 2.f/3;   // (1)
//
//	return (val + log_2);
//}

//M_INLINE
//float fast_pow (const float val1, const float val2)
//{
//	return _fast_pow2(val2 * _fast_log2(val1));
//}

// ------------------------------------------------------------------------ //
// Vec2
// ------------------------------------------------------------------------ //

M_INLINE
float Vec2::Dot(const Vec2 &a, const Vec2 &b)
{
	return (a.x * b.x) + (a.y * b.y);
}

M_INLINE
Vec2 Vec2::Cross(const Vec2 &a)
{
	return Vec2(a.y, -a.x);
}

M_INLINE
Vec2 Vec2::Lerp(const Vec2 &a, const Vec2 &b, float f)
{
	return Vec2(a.x + (b.x - a.x) * f, a.y + (b.y - a.y) * f);
}

// assignment operators
// ----------------------------------------------------------------------------

M_INLINE
void Vec2::operator += (const Vec2 &v)
{
	x += v.x;
	y += v.y;
}

M_INLINE
void Vec2::operator -= (const Vec2 &v)
{
	x -= v.x;
	y -= v.y;
}

M_INLINE
void Vec2::operator *= (const Vec2 &v)
{
	x *= v.x;
	y *= v.y;
}

M_INLINE
void Vec2::operator /= (const Vec2 &v)
{
	x /= v.x;
	y /= v.y;
}

M_INLINE
void Vec2::operator += (float f)
{
	x += f;
	y += f;
}

M_INLINE
void Vec2::operator -= (float f)
{
	float	fInv = 1.f / f;
	x -= fInv;
	y -= fInv;
}

M_INLINE
void Vec2::operator *= (float f)
{
	x *= f;
	y *= f;
}

M_INLINE
void Vec2::operator /= (float f)
{
	float	fInv = 1.f / f;
	x *= fInv;
	y *= fInv;
}

// unary operators

M_INLINE
Vec2 Vec2::operator + () const
{
	return *this;
}

M_INLINE
Vec2 Vec2::operator - () const
{
	return Vec2 (-x, -y);
}

// binary operators

M_INLINE
Vec2 Vec2::operator + (const Vec2 &v) const
{
	return Vec2 (x + v.x, y + v.y);
}

M_INLINE
Vec2 Vec2::operator - (const Vec2 &v) const
{
	return Vec2 (x - v.x, y - v.y);
}

M_INLINE
Vec2 Vec2::operator * (const Vec2 &v) const
{
	return Vec2 (x * v.x, y * v.y);
}

M_INLINE
Vec2 Vec2::operator / (const Vec2 &v) const
{
	return Vec2 (x / v.x, y / v.y);
}

M_INLINE
Vec2 Vec2::operator + (float f) const
{
	return Vec2 (x + f, y + f);
}

M_INLINE
Vec2 Vec2::operator - (float f) const
{
	return Vec2 (x - f, y - f);
}

M_INLINE
Vec2 Vec2::operator * (float f) const
{
	return Vec2 (x * f, y * f);
}

M_INLINE
Vec2 Vec2::operator / (float f) const
{
	float	fInv = 1.f / f;
	return Vec2 (x * fInv, y * fInv);
}

M_INLINE
Vec2 operator + (float f, const Vec2 &v)
{
	return Vec2 (f + v.x, f + v.y);
}

M_INLINE
Vec2 operator - (float f, const Vec2 &v)
{
	return Vec2 (f - v.x, f - v.y);
}

M_INLINE
Vec2 operator * (float f, const Vec2 &v)
{
	return Vec2 (f * v.x, f * v.y);
}

M_INLINE
bool Vec2::operator == (const Vec2 &v) const
{
	return ((v.x == x) && (v.y == y));
}

M_INLINE
bool Vec2::operator != (const Vec2 &v) const
{
	return ((v.x != x) || (v.y != y));
}

M_INLINE
float Vec2::Length() const
{
	return sqrtf(x * x + y * y);
}

M_INLINE
float Vec2::Length2() const
{
	return (x * x + y * y);
}

M_INLINE
float Vec2::Normalize()
{
	float length = sqrtf(x * x + y * y);
	if (length != 0.f) {
		float f = 1.f / length;
		x *= f;
		y *= f;
	}
	return length;
}

// ------------------------------------------------------------------------ //
// Vec3
// ------------------------------------------------------------------------ //

// static functions

M_INLINE
float Vec3::Dot(const Vec3 &a, const Vec3 &b)
{
	return (a.x * b.x) + (a.y * b.y) + (a.z * b.z);
}

M_INLINE
Vec3 Vec3::Cross(const Vec3 &a, const Vec3 &b)
{
	return Vec3((a.y * b.z) - (a.z * b.y),
				(a.z * b.x) - (a.x * b.z),
				(a.x * b.y) - (a.y * b.x));
}

M_INLINE
Vec3 Vec3::Lerp(const Vec3 &a, const Vec3 &b, float f)
{
	return Vec3(a.x + (b.x - a.x) * f,
				a.y + (b.y - a.y) * f,
				a.z + (b.z - a.z) * f);
}

M_INLINE
Vec3 Vec3::Reflect(const Vec3 &v, const Vec3 &n)
{
	return v - 2 * Dot(v, n) * n;
}

// assignment operators

M_INLINE
void Vec3::operator += (const Vec3 &v)
{
	x += v.x;
	y += v.y;
	z += v.z;
}

M_INLINE
void Vec3::operator -= (const Vec3 &v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;
}

M_INLINE
void Vec3::operator *= (const Vec3 &v)
{
	x *= v.x;
	y *= v.y;
	z *= v.z;
}

M_INLINE
void Vec3::operator /= (const Vec3 &v)
{
	x /= v.x;
	y /= v.y;
	z /= v.z;
}

M_INLINE
void Vec3::operator += (float f)
{
	x += f;
	y += f;
	z += f;
}

M_INLINE
void Vec3::operator -= (float f)
{
	x -= f;
	y -= f;
	z -= f;
}

M_INLINE
void Vec3::operator *= (float f)
{
	x *= f;
	y *= f;
	z *= f;
}

M_INLINE
void Vec3::operator /= (float f)
{
	float fInv = 1.f / f;
	x *= fInv;
	y *= fInv;
	z *= fInv;
}

// unary operators

M_INLINE
Vec3 Vec3::operator + () const
{
	return *this;
}

M_INLINE
Vec3 Vec3::operator - () const
{
	return Vec3(-x, -y, -z);
}

// binary operators

M_INLINE
Vec3 Vec3::operator + (const Vec3 &other) const
{
	return Vec3 (x + other.x, y + other.y, z + other.z);
}

M_INLINE
Vec3 Vec3::operator - (const Vec3 &other) const
{
	return Vec3 (x - other.x, y - other.y, z - other.z);
}

M_INLINE
Vec3 Vec3::operator * (const Vec3 &other) const
{
	return Vec3 (x * other.x, y * other.y, z * other.z);
}

M_INLINE
Vec3 Vec3::operator / (const Vec3 &other) const
{
	return Vec3 (x / other.x, y / other.y, z / other.z);
}

M_INLINE
Vec3 Vec3::operator + (float f) const
{
	return Vec3 (x + f, y + f, z + f);
}

M_INLINE
Vec3 Vec3::operator - (float f) const
{
	return Vec3 (x - f, y - f, z - f);
}

M_INLINE
Vec3 Vec3::operator * (float f) const
{
	return Vec3 (x * f, y * f, z * f);
}

M_INLINE
Vec3 Vec3::operator / (float f) const
{
	float fInv = 1.f / f;
	return Vec3(x * fInv, y * fInv, z * fInv);
}

M_INLINE
Vec3 operator + (float f, const Vec3 &v)
{
	return Vec3(f + v.x, f + v.y, f + v.z);
}

M_INLINE
Vec3 operator - (float f, const Vec3 &v)
{
	return Vec3(f - v.x, f - v.y, f - v.z);
}

M_INLINE
Vec3 operator * (float f, const Vec3 &v)
{
	return Vec3(f * v.x, f * v.y, f * v.z);
}

M_INLINE
bool Vec3::operator == (const Vec3 &v) const
{
	return ((v.x == x) && (v.y == y) && (v.z == z));
}

M_INLINE
bool Vec3::operator != (const Vec3 &v) const
{
	return ((v.x != x) || (v.y != y) || (v.z != z));
}

M_INLINE
float Vec3::Length() const
{
	return sqrtf(x * x + y * y + z * z);
}

M_INLINE
float Vec3::Length2() const
{
	return (x * x + y * y + z * z);
}

M_INLINE
float Vec3::Normalize()
{
	float length = sqrtf(x * x + y * y + z * z);
	if (length != 0.f) {
		float f = 1.f / length;
		x *= f;
		y *= f;
		z *= f;
	}
	return length;
}

M_INLINE
Vec3 Vec3::NormalFromColor(const ColorF &clr)
{
	return Vec3((clr.r - 0.5f) * 2, (clr.g - 0.5f) * 2, (clr.b - 0.5f) * 2);
}

// ------------------------------------------------------------------------ //
// Vec4
// ------------------------------------------------------------------------ //

M_INLINE
float Vec4::Dot(const Vec4 &a, const Vec4 &b)
{
	return (a.x * b.x) + (a.y * b.y) + (a.z * b.z) + (a.w * b.w);
}

M_INLINE
Vec4 Vec4::Cross(const Vec4 &a, const Vec4 &b, const Vec4 &c)
{
	Vec4 result;
	// XXX can this be improved? Look at assembly.
//#define ROW(i)       a[i], b[i], c[i]
//#define DET(i,j,k)   dot(Vec3(ROW(i)), cross(Vec3(ROW(j)), Vec3(ROW(k))))

	result.x =  Vec3::Dot(Vec3(a.y, b.y, c.y), Vec3::Cross(Vec3(a.z, b.z, c.z), Vec3(a.w, b.w, c.w)));
	result.y = -Vec3::Dot(Vec3(a.x, b.x, c.x), Vec3::Cross(Vec3(a.z, b.z, c.z), Vec3(a.w, b.w, c.w)));
	result.z =  Vec3::Dot(Vec3(a.x, b.x, c.x), Vec3::Cross(Vec3(a.y, b.y, c.y), Vec3(a.w, b.w, c.w)));
	result.w = -Vec3::Dot(Vec3(a.x, b.x, c.x), Vec3::Cross(Vec3(a.y, b.y, c.y), Vec3(a.z, b.z, c.z)));
	//result.x =  DET(1,2,3);
	//result.y = -DET(0,2,3);
	//result.z =  DET(0,1,3);
	//result.w = -DET(0,1,2);

	return result;

//#undef ROW
//#undef DET
}

M_INLINE
Vec4 Vec4::Lerp(const Vec4 &a, const Vec4 &b, float f)
{
	return Vec4(a.x + (b.x - a.x) * f,
				a.y + (b.y - a.y) * f,
				a.z + (b.z - a.z) * f,
				a.w + (b.w - a.w) * f);
}

// ------------------------------------------------------------------------ //
// iPoint
// ------------------------------------------------------------------------ //

M_INLINE
bool iPoint::operator == (const iPoint &p) const
{
	return ((p.x == x) && (p.y == y));
}

M_INLINE
bool iPoint::operator != (const iPoint &p) const
{
	return ((p.x != x) || (p.y != y));
}

M_INLINE
iPoint& iPoint::operator += (const iPoint &p)
{
	x += p.x;
	y += p.y;
	return *this;
}

M_INLINE
iPoint& iPoint::operator -= (const iPoint &p)
{
	x -= p.x;
	y -= p.y;
	return *this;
}

M_INLINE
iPoint iPoint::operator + (const iPoint &p) const
{
	return iPoint(x + p.x, y + p.y);
}

M_INLINE
iPoint iPoint::operator - (const iPoint &p) const
{
	return iPoint(x - p.x, y - p.y);
}

M_INLINE
iPoint iPoint::operator - () const
{
	return iPoint(-x, -y);
}

// ------------------------------------------------------------------------ //
// Color
// ------------------------------------------------------------------------ //

M_INLINE
ColorB::operator ColorW () const
{
	ColorW c;
	c.b = b << 8;
	c.g = g << 8;
	c.r = r << 8;
	c.a = a << 8;
	return c;
}

M_INLINE
ColorB::operator ColorF () const
{
	ColorF c;
	c.b = B2F(b);
	c.g = B2F(g);
	c.r = B2F(r);
	c.a = B2F(a);
	return c;
}

// assignment operators

M_INLINE
void ColorB::operator += (const ColorB &c)
{
	int res_r, res_g, res_b, res_a;
	res_r = r + c.r;
	res_g = g + c.g;
	res_b = b + c.b;
	res_a = a + c.a;
	r = (res_r < 255) ? res_r : 255;
	g = (res_g < 255) ? res_g : 255;
	b = (res_b < 255) ? res_b : 255;
	a = (res_a < 255) ? res_a : 255;
}

M_INLINE
void ColorB::operator -= (const ColorB &c)
{
	r = (r > c.r) ? (r - c.r) : 0;
	g = (g > c.g) ? (g - c.g) : 0;
	b = (b > c.b) ? (b - c.b) : 0;
	a = (a > c.a) ? (a - c.a) : 0;
}

M_INLINE
void ColorB::operator *= (float scale)
{
	r = (byte)(r * scale);
	g = (byte)(g * scale);
	b = (byte)(b * scale);
	a = (byte)(a * scale);
}

M_INLINE
void ColorB::operator *= (ColorB c)
{
	float	scale = 1.f / 255.f;
	r = (byte) ((float)r * (float)c.r * scale);
	g = (byte) ((float)g * (float)c.g * scale);
	b = (byte) ((float)b * (float)c.b * scale);
	a = (byte) ((float)a * (float)c.a * scale);
}

M_INLINE
void ColorB::operator /= (float k)
{
	float	scale = 1.f / k;
	r = (byte)(r * scale);
	g = (byte)(g * scale);
	b = (byte)(b * scale);
	a = (byte)(a * scale);
}

// binary operators

M_INLINE
ColorB ColorB::operator + (const ColorB &c) const
{
	ColorB ret;
	int		res_r, res_g, res_b, res_a;
	res_r = r + c.r;
	res_g = g + c.g;
	res_b = b + c.b;
	res_a = a + c.a;
	ret.r = (res_r < 255) ? res_r : 255;
	ret.g = (res_g < 255) ? res_g : 255;
	ret.b = (res_b < 255) ? res_b : 255;
	ret.a = (res_a < 255) ? res_a : 255;
	return ret;
}

M_INLINE
ColorB ColorB::operator - (const ColorB &c) const
{
	ColorB ret;
	ret.r = (r > c.r) ? (r - c.r) : 0;
	ret.g = (g > c.g) ? (g - c.g) : 0;
	ret.b = (b > c.b) ? (b - c.b) : 0;
	ret.a = (a > c.a) ? (a - c.a) : 0;
	return ret;
}

M_INLINE
ColorB ColorB::operator * (float scale) const
{
	ColorB ret;
	ret.r = (byte)(r * scale);
	ret.g = (byte)(g * scale);
	ret.b = (byte)(b * scale);
	ret.a = (byte)(a * scale);
	return ret;
}

M_INLINE
ColorB ColorB::operator * (ColorB c) const
{
	ColorB ret;
	ret.r = (byte)(((int)r * (int)c.r) >> 8);
	ret.g = (byte)(((int)g * (int)c.g) >> 8);
	ret.b = (byte)(((int)b * (int)c.b) >> 8);
	ret.a = (byte)(((int)a * (int)c.a) >> 8);
	return ret;
}

M_INLINE
ColorB ColorB::operator / (float k) const
{
	float	scale = 1.f / k;
	ColorB ret;
	ret.r = (byte)(r * scale);
	ret.g = (byte)(g * scale);
	ret.b = (byte)(b * scale);
	ret.a = (byte)(a * scale);
	return ret;
}

// functions

M_INLINE
Color ColorB::Lerp(const ColorB &from, const ColorB &to, float k)
{
	//if (*((int *)&from) == *((int *)&to)) return from;
	if (from == to) return from;

	int r = from.r + (int)((to.r - from.r) * k);
	int g = from.g + (int)((to.g - from.g) * k);
	int b = from.b + (int)((to.b - from.b) * k);
	int a = from.a + (int)((to.a - from.a) * k);
	return ColorB(r, g, b, a);
	//	if (r < 0) r = 0; else if (r > 0xff) r = 0xff;
	//	if (g < 0) g = 0; else if (g > 0xff) g = 0xff;
	//	if (b < 0) b = 0; else if (b > 0xff) b = 0xff;
	//	if (a < 0) a = 0; else if (a > 0xff) a = 0xff;
}

// ------------------------------------------------------------------------ //

M_INLINE
ColorW::operator ColorB () const
{
	ColorB c;
	c.b = b >> 8;
	c.g = g >> 8;
	c.r = r >> 8;
	c.a = a >> 8;
	return c;
}

M_INLINE
ColorW::operator ColorF () const
{
	ColorF c;
	c.b = W2F(b);
	c.g = W2F(g);
	c.r = W2F(r);
	c.a = W2F(a);
	return c;
}

M_INLINE
void ColorW::operator += (const ColorW &c)
{
	int		res_r, res_g, res_b, res_a;
	res_r = clamp<int>(r + c.r, 0, 65535);
	res_g = clamp<int>(g + c.g, 0, 65535);
	res_b = clamp<int>(b + c.b, 0, 65535);
	res_a = clamp<int>(a + c.a, 0, 65535);
	r = res_r; g = res_g; b = res_b; a = res_a;
}

M_INLINE
void ColorW::operator -= (const ColorW &c)
{
	int		res_r, res_g, res_b, res_a;
	res_r = clamp<int>(r - c.r, 0, 65535);
	res_g = clamp<int>(g - c.g, 0, 65535);
	res_b = clamp<int>(b - c.b, 0, 65535);
	res_a = clamp<int>(a - c.a, 0, 65535);
	r = res_r; g = res_g; b = res_b; a = res_a;
}

M_INLINE
void ColorW::operator *= (float scale)
{
	r = static_cast<short>(r * scale);
	g = static_cast<short>(g * scale);
	b = static_cast<short>(b * scale);
	a = static_cast<short>(a * scale);
}

M_INLINE
void ColorW::operator *= (ColorW c)
{
	r = F2W(W2F(r) * W2F(c.r));
	g = F2W(W2F(g) * W2F(c.g));
	b = F2W(W2F(b) * W2F(c.b));
	a = F2W(W2F(a) * W2F(c.a));
}

M_INLINE
void ColorW::operator /= (float k)
{
	float	scale = 1.f / k;
	r = static_cast<short>(r * scale);
	g = static_cast<short>(g * scale);
	b = static_cast<short>(b * scale);
	a = static_cast<short>(a * scale);
}

// binary operators

M_INLINE
ColorW ColorW::operator + (const ColorW &c) const
{
	return ColorW(
			clamp<int>(r + c.r, 0, 65535),
			clamp<int>(g + c.g, 0, 65535),
			clamp<int>(b + c.b, 0, 65535),
			clamp<int>(a + c.a, 0, 65535)
		);
}

M_INLINE
ColorW ColorW::operator - (const ColorW &c) const
{
	return ColorW(
		clamp<int>(r - c.r, 0, 65535),
		clamp<int>(g - c.g, 0, 65535),
		clamp<int>(b - c.b, 0, 65535),
		clamp<int>(a - c.a, 0, 65535)
		);
}

M_INLINE
ColorW ColorW::operator * (float scale) const
{
	return ColorW(
		static_cast<ushort>(r * scale),
		static_cast<ushort>(g * scale),
		static_cast<ushort>(b * scale),
		static_cast<ushort>(a * scale)
		);
}

M_INLINE
ColorW ColorW::operator * (ColorW c) const
{
	return ColorW(F2W(W2F(r)*W2F(c.r)), F2W(W2F(g)*W2F(c.g)), F2W(W2F(b)*W2F(c.b)), F2W(W2F(a)*W2F(c.a)));
}

M_INLINE
ColorW ColorW::operator / (float k) const
{
	float	scale = 1.f / k;
	return ColorW (
		static_cast<ushort>(r * scale),
		static_cast<ushort>(g * scale),
		static_cast<ushort>(b * scale),
		static_cast<ushort>(a * scale)
		);
}

M_INLINE
ColorW ColorW::Lerp(const ColorW &from, const ColorW &to, float k)
{
	if (from == to) return from;

	int r = from.r + (int)((to.r - from.r) * k);
	int g = from.g + (int)((to.g - from.g) * k);
	int b = from.b + (int)((to.b - from.b) * k);
	int a = from.a + (int)((to.a - from.a) * k);
	return ColorW(r, g, b, a);
}

// ------------------------------------------------------------------------ //

M_INLINE
ColorF::operator ColorB () const
{
	ColorB c;
	c.b = F2B(clamp(b, 0.f, 1.f));
	c.g = F2B(clamp(g, 0.f, 1.f));
	c.r = F2B(clamp(r, 0.f, 1.f));
	c.a = F2B(clamp(a, 0.f, 1.f));
	return c;
}

M_INLINE
ColorF::operator ColorW () const
{
	ColorF c;
	c.b = F2W(b);
	c.g = F2W(g);
	c.r = F2W(r);
	c.a = F2W(a);
	return c;
}


M_INLINE
void ColorF::operator += (const ColorF &c)
{
	r += c.r; g += c.g; b += c.b; a += c.a;
}

M_INLINE
void ColorF::operator -= (const ColorF &c)
{
	r -= c.r; g -= c.g; b -= c.b; a -= c.a;
}

M_INLINE
void ColorF::operator *= (float scale)
{
	r *= scale;	g *= scale; b *= scale; a *= scale;
}

M_INLINE
void ColorF::operator *= (ColorF c)
{
	r *= c.r; g *= c.g; b *= c.b; a *= c.a;
}

M_INLINE
void ColorF::operator /= (float k)
{
	float	scale = 1.f / k;
	r *= scale;	g *= scale; b *= scale; a *= scale;
}

// binary operators

M_INLINE
ColorF ColorF::operator + (const ColorF &c) const
{
	return ColorF(r + c.r, g + c.g, b + c.b, a + c.a);
}

M_INLINE
ColorF ColorF::operator - (const ColorF &c) const
{
	return ColorF(r - c.r, g - c.g, b - c.b, a - c.a);
}

M_INLINE
ColorF ColorF::operator * (float scale) const
{
	return ColorF(r * scale, g * scale, b * scale, a * scale);
}

M_INLINE
ColorF ColorF::operator * (ColorF c) const
{
	return ColorF(r * c.r, g * c.g, b * c.b, a * c.a);
}

M_INLINE
ColorF ColorF::operator / (float k) const
{
	float	scale = 1.f / k;
	return ColorF(r * scale, g * scale, b * scale, a * scale);
}

// functions

M_INLINE
ColorF ColorF::Lerp(const ColorF &from, const ColorF &to, float k)
{
	if (from == to) return from;

	float b = from.b + (to.b - from.b) * k;
	float g = from.g + (to.g - from.g) * k;
	float r = from.r + (to.r - from.r) * k;
	float a = from.a + (to.a - from.a) * k;
	return ColorF(r, g, b, a);
}

M_INLINE
ColorF ColorF::FromHSL(float fHue, float fSat, float fLight)
{
	if (fSat == 0) return ColorF(fLight, fLight, fLight, 1.f);

	ColorF res(0.f, 0.f, 0.f, 1.f);
	
	float fTemp2 = (fLight < 0.5f) ? fLight * (1 + fSat) : (fLight + fSat) - (fLight * fSat);
	float fTemp1 = 2 * fLight - fTemp2;
	
	float fTempR = fHue + 1.f / 3.f;
	if (fTempR > 1.f) fTempR -= 1.f;
	float fTempG = fHue;
	float fTempB = fHue - 1.f / 3.f;
	if (fTempB < 0.f) fTempB += 1.f;
	
	if (6 * fTempR < 1.f)
		res.r = fTemp1 + (fTemp2 - fTemp1) * 6 * fTempR;
	else if (2 * fTempR < 1.f)
		res.r = fTemp2;
	else if (3 * fTempR < 2)
		res.r = fTemp1 + (fTemp2 - fTemp1) * (2.f / 3.f - fTempR) * 6.f;
	else
		res.r = fTemp1;

	if (6 * fTempG < 1.f)
		res.g = fTemp1 + (fTemp2 - fTemp1) * 6 * fTempG;
	else if (2 * fTempG < 1.f)
		res.g = fTemp2;
	else if (3 * fTempG < 2)
		res.g = fTemp1 + (fTemp2 - fTemp1) * (2.f / 3.f - fTempG) * 6.f;
	else
		res.g = fTemp1;

	if (6 * fTempB < 1.f)
		res.b = fTemp1 + (fTemp2 - fTemp1) * 6 * fTempB;
	else if (2 * fTempB < 1.f)
		res.b = fTemp2;
	else if (3 * fTempB < 2)
		res.b = fTemp1 + (fTemp2 - fTemp1) * (2.f / 3.f - fTempB) * 6.f;
	else
		res.b = fTemp1;
	
	return res;
}


M_INLINE
bool ColorF::IsSimilarTo(const ColorF &ref, float fEpsilon) const
{
	float dr = fabs(r - ref.r);
	float dg = fabs(g - ref.g);
	float db = fabs(b - ref.b);
	float da = fabs(a - ref.a);
	return dr + dg + db + da < fEpsilon;
}



// ------------------------------------------------------------------------ //
// Matrix
// ------------------------------------------------------------------------ //

M_INLINE
Matrix::Matrix(const Matrix & mat)
{
	::memcpy(m, mat.m, sizeof(m));
}

M_INLINE
Matrix::Matrix(const float mat[16])
{
	::memcpy(m, mat, sizeof(m));
}

M_INLINE
Matrix::Matrix( float f11, float f12, float f13, float f14,
				float f21, float f22, float f23, float f24,
				float f31, float f32, float f33, float f34,
				float f41, float f42, float f43, float f44)
{
	m11 = f11; m12 = f12; m13 = f13; m14 = f14;
	m21 = f21; m22 = f22; m23 = f23; m24 = f24;
	m31 = f31; m32 = f32; m33 = f33; m34 = f34;
	m41 = f41; m42 = f42; m43 = f43; m44 = f44;
}

M_INLINE
void MatrixMultiply(Matrix & out, const Matrix & a, const Matrix & b)
{
	Matrix mat;
	Matrix & target = (&out != &a && &out != &b) ? out : mat;
	//Matrix & target = out;

	target.m11 = a.m11*b.m11 + a.m12*b.m21 + a.m13*b.m31 + a.m14*b.m41;
	target.m12 = a.m11*b.m12 + a.m12*b.m22 + a.m13*b.m32 + a.m14*b.m42;
	target.m13 = a.m11*b.m13 + a.m12*b.m23 + a.m13*b.m33 + a.m14*b.m43;
	target.m14 = a.m11*b.m14 + a.m12*b.m24 + a.m13*b.m34 + a.m14*b.m44;

	target.m21 = a.m21*b.m11 + a.m22*b.m21 + a.m23*b.m31 + a.m24*b.m41;
	target.m22 = a.m21*b.m12 + a.m22*b.m22 + a.m23*b.m32 + a.m24*b.m42;
	target.m23 = a.m21*b.m13 + a.m22*b.m23 + a.m23*b.m33 + a.m24*b.m43;
	target.m24 = a.m21*b.m14 + a.m22*b.m24 + a.m23*b.m34 + a.m24*b.m44;

	target.m31 = a.m31*b.m11 + a.m32*b.m21 + a.m33*b.m31 + a.m34*b.m41;
	target.m32 = a.m31*b.m12 + a.m32*b.m22 + a.m33*b.m32 + a.m34*b.m42;
	target.m33 = a.m31*b.m13 + a.m32*b.m23 + a.m33*b.m33 + a.m34*b.m43;
	target.m34 = a.m31*b.m14 + a.m32*b.m24 + a.m33*b.m34 + a.m34*b.m44;

	target.m41 = a.m41*b.m11 + a.m42*b.m21 + a.m43*b.m31 + a.m44*b.m41;
	target.m42 = a.m41*b.m12 + a.m42*b.m22 + a.m43*b.m32 + a.m44*b.m42;
	target.m43 = a.m41*b.m13 + a.m42*b.m23 + a.m43*b.m33 + a.m44*b.m43;
	target.m44 = a.m41*b.m14 + a.m42*b.m24 + a.m43*b.m34 + a.m44*b.m44;

	if (&out != &target) out = target;
}

// assignment operators

M_INLINE
Matrix& Matrix::operator *= (const Matrix& mat)
{
	MatrixMultiply(*this, *this, mat);
	return *this;
}

M_INLINE
Matrix& Matrix::operator += (const Matrix& mat)
{
	m11 += mat.m11; m12 += mat.m12; m13 += mat.m13; m14 += mat.m14;
	m21 += mat.m21; m22 += mat.m22; m23 += mat.m23; m24 += mat.m24;
	m31 += mat.m31; m32 += mat.m32; m33 += mat.m33; m34 += mat.m34;
	m41 += mat.m41; m42 += mat.m42; m43 += mat.m43; m44 += mat.m44;
	return *this;
}

M_INLINE
Matrix& Matrix::operator -= (const Matrix& mat)
{
	m11 -= mat.m11; m12 -= mat.m12; m13 -= mat.m13; m14 -= mat.m14;
	m21 -= mat.m21; m22 -= mat.m22; m23 -= mat.m23; m24 -= mat.m24;
	m31 -= mat.m31; m32 -= mat.m32; m33 -= mat.m33; m34 -= mat.m34;
	m41 -= mat.m41; m42 -= mat.m42; m43 -= mat.m43; m44 -= mat.m44;
	return *this;
}

M_INLINE
Matrix& Matrix::operator *= (float f)
{
	m11 *= f; m12 *= f; m13 *= f; m14 *= f;
	m21 *= f; m22 *= f; m23 *= f; m24 *= f;
	m31 *= f; m32 *= f; m33 *= f; m34 *= f;
	m41 *= f; m42 *= f; m43 *= f; m44 *= f;
	return *this;
}

M_INLINE
Matrix& Matrix::operator /= (float f)
{
	float fInv = 1.f / f;
	m11 *= fInv; m12 *= fInv; m13 *= fInv; m14 *= fInv;
	m21 *= fInv; m22 *= fInv; m23 *= fInv; m24 *= fInv;
	m31 *= fInv; m32 *= fInv; m33 *= fInv; m34 *= fInv;
	m41 *= fInv; m42 *= fInv; m43 *= fInv; m44 *= fInv;
	return *this;
}

// unary operators

M_INLINE
Matrix Matrix::operator + () const
{
	return *this;
}

M_INLINE
Matrix Matrix::operator - () const
{
	return	Matrix( -m11, -m12, -m13, -m14,
					-m21, -m22, -m23, -m24,
					-m31, -m32, -m33, -m34,
					-m41, -m42, -m43, -m44 );
}

// binary operators

M_INLINE
Matrix Matrix::operator * (const Matrix& mat) const
{
	Matrix matT;
	MatrixMultiply(matT, *this, mat);
	return matT;
}

M_INLINE
Matrix Matrix::operator + (const Matrix& mat) const
{
	return	Matrix( m11 + mat.m11, m12 + mat.m12, m13 + mat.m13, m14 + mat.m14,
					m21 + mat.m21, m22 + mat.m22, m23 + mat.m23, m24 + mat.m24,
					m31 + mat.m31, m32 + mat.m32, m33 + mat.m33, m34 + mat.m34,
					m41 + mat.m41, m42 + mat.m42, m43 + mat.m43, m44 + mat.m44 );
}

M_INLINE
Matrix Matrix::operator - (const Matrix& mat) const
{
	return	Matrix( m11 - mat.m11, m12 - mat.m12, m13 - mat.m13, m14 - mat.m14,
					m21 - mat.m21, m22 - mat.m22, m23 - mat.m23, m24 - mat.m24,
					m31 - mat.m31, m32 - mat.m32, m33 - mat.m33, m34 - mat.m34,
					m41 - mat.m41, m42 - mat.m42, m43 - mat.m43, m44 - mat.m44 );
}

M_INLINE
Matrix Matrix::operator * (float f) const
{
	return	Matrix( m11 * f, m12 * f, m13 * f, m14 * f,
					m21 * f, m22 * f, m23 * f, m24 * f,
					m31 * f, m32 * f, m33 * f, m34 * f,
					m41 * f, m42 * f, m43 * f, m44 * f );
}

M_INLINE
Matrix Matrix::operator / (float f) const
{
	float fInv = 1.f / f;
	return	Matrix( m11 * fInv, m12 * fInv, m13 * fInv, m14 * fInv,
					m21 * fInv, m22 * fInv, m23 * fInv, m24 * fInv,
					m31 * fInv, m32 * fInv, m33 * fInv, m34 * fInv,
					m41 * fInv, m42 * fInv, m43 * fInv, m44 * fInv );
}

M_INLINE
Matrix operator * (float f, const Matrix& mat)
{
	return	Matrix( f * mat.m11, f * mat.m12, f * mat.m13, f * mat.m14,
					f * mat.m21, f * mat.m22, f * mat.m23, f * mat.m24,
					f * mat.m31, f * mat.m32, f * mat.m33, f * mat.m34,
					f * mat.m41, f * mat.m42, f * mat.m43, f * mat.m44 );
}

M_INLINE
bool Matrix::operator == (const Matrix& mat) const
{
	return 0 == ::memcmp(m, &mat.m, sizeof(m));
}

M_INLINE
bool Matrix::operator != (const Matrix& mat) const
{
	return 0 != ::memcmp(m, &mat.m, sizeof(m));
}

//#define	MatrixDet3(mat, iRow1, iRow2, iRow3, iCol1, iCol2, iCol3) (mat.m[iRow1][iCol1] * (mat.m[iRow2][iCol2]*mat.m[iRow3][iCol3] - mat.m[iRow3][iCol2]*mat.m[iRow2][iCol3]) - mat.m[iRow1][iCol2] * (mat.m[iRow2][iCol1]*mat.m[iRow3][iCol3] - mat.m[iRow3][iCol1]*mat.m[iRow2][iCol3]) + mat.m[iRow1][iCol3] * (mat.m[iRow2][iCol1]*mat.m[iRow3][iCol2] - mat.m[iRow3][iCol1]*mat.m[iRow2][iCol2]))
//
//M_INLINE
//void Matrix::Inverse (const Matrix& mat)
//{
//	Matrix matTmp;
//	matTmp.m11 =  MatrixDet3(mat, 1, 2, 3, 1, 2, 3);
//	matTmp.m21 = -MatrixDet3(mat, 1, 2, 3, 0, 2, 3);
//	matTmp.m31 =  MatrixDet3(mat, 1, 2, 3, 0, 1, 3);
//	matTmp.m41 = -MatrixDet3(mat, 1, 2, 3, 0, 1, 2);
//	matTmp.m12 = -MatrixDet3(mat, 0, 2, 3, 1, 2, 3);
//	matTmp.m22 =  MatrixDet3(mat, 0, 2, 3, 0, 2, 3);
//	matTmp.m32 = -MatrixDet3(mat, 0, 2, 3, 0, 1, 3);
//	matTmp.m42 =  MatrixDet3(mat, 0, 2, 3, 0, 1, 2);
//	matTmp.m13 =  MatrixDet3(mat, 0, 1, 3, 1, 2, 3);
//	matTmp.m23 = -MatrixDet3(mat, 0, 1, 3, 0, 2, 3);
//	matTmp.m33 =  MatrixDet3(mat, 0, 1, 3, 0, 1, 3);
//	matTmp.m43 = -MatrixDet3(mat, 0, 1, 3, 0, 1, 2);
//	matTmp.m14 = -MatrixDet3(mat, 0, 1, 2, 1, 2, 3);
//	matTmp.m24 =  MatrixDet3(mat, 0, 1, 2, 0, 2, 3);
//	matTmp.m34 = -MatrixDet3(mat, 0, 1, 2, 0, 1, 3);
//	matTmp.m44 =  MatrixDet3(mat, 0, 1, 2, 0, 1, 2);
//	float det = mat.m11 * matTmp.m11 + mat.m12 * matTmp.m21 + mat.m13 * matTmp.m31 + mat.m14 * matTmp.m41;
//	if (fabsf(det) < 1e-5) return;
//	*this = matTmp / det;
//}

// ------------------------------------------------------------------------ //
// Bounding Box
// ------------------------------------------------------------------------ //

M_INLINE
bool sBBox::IsEmpty() const
{
	return vMins.x >= vMaxs.x || vMins.y >= vMaxs.y || vMins.z >= vMaxs.z;
}

M_INLINE
Vec3 sBBox::Center() const
{
	return (vMins + vMaxs) * 0.5f;
}

M_INLINE
Vec3 sBBox::Size() const
{
	Vec3 vSize = vMaxs - vMins;
	if (vSize.x < 0) vSize.x = 0;
	if (vSize.y < 0) vSize.y = 0;
	if (vSize.z < 0) vSize.z = 0;
	return vSize;
}

M_INLINE
void sBBox::ClearBounds()
{
	vMins.x = vMins.y = vMins.z = FLT_MAX;
	vMaxs.x = vMaxs.y = vMaxs.z = -FLT_MAX;
}

M_INLINE
void sBBox::AddToBounds(const Vec3 & v)
{
	if (vMins.x > v.x) vMins.x = v.x;
	if (vMaxs.x < v.x) vMaxs.x = v.x;
	if (vMins.y > v.y) vMins.y = v.y;
	if (vMaxs.y < v.y) vMaxs.y = v.y;
	if (vMins.z > v.z) vMins.z = v.z;
	if (vMaxs.z < v.z) vMaxs.z = v.z;
}

M_INLINE
void sBBox::AddToBounds(const sBBox & box)
{
	if (vMins.x > box.vMins.x) vMins.x = box.vMins.x;
	if (vMaxs.x < box.vMaxs.x) vMaxs.x = box.vMaxs.x;
	if (vMins.y > box.vMins.y) vMins.y = box.vMins.y;
	if (vMaxs.y < box.vMaxs.y) vMaxs.y = box.vMaxs.y;
	if (vMins.z > box.vMins.z) vMins.z = box.vMins.z;
	if (vMaxs.z < box.vMaxs.z) vMaxs.z = box.vMaxs.z;
}

M_INLINE
void sBBox::LimitBy(const sBBox box)
{
	if (vMins.x < box.vMins.x) vMins.x = box.vMins.x;
	if (vMins.y < box.vMins.y) vMins.y = box.vMins.y;
	if (vMins.z < box.vMins.z) vMins.z = box.vMins.z;
	if (vMaxs.x > box.vMaxs.x) vMaxs.x = box.vMaxs.x;
	if (vMaxs.y > box.vMaxs.y) vMaxs.y = box.vMaxs.y;
	if (vMaxs.z > box.vMaxs.z) vMaxs.z = box.vMaxs.z;
}

M_INLINE
bool sBBox::IsIntersect(const sBBox &b1, const sBBox &b2)
{
	if (b1.vMins.x > b2.vMaxs.x) return false;
	if (b1.vMaxs.x < b2.vMins.x) return false;
	if (b1.vMins.y > b2.vMaxs.y) return false;
	if (b1.vMaxs.y < b2.vMins.y) return false;
	if (b1.vMins.z > b2.vMaxs.z) return false;
	if (b1.vMaxs.z < b2.vMins.z) return false;
	return true;
}

M_INLINE
bool sBBox::IsInside(const sBBox &bigBox) const
{
	if (vMins.x <= bigBox.vMins.x) return false;
	if (vMins.y <= bigBox.vMins.y) return false;
	if (vMins.z <= bigBox.vMins.z) return false;
	if (vMaxs.x >= bigBox.vMaxs.x) return false;
	if (vMaxs.y >= bigBox.vMaxs.y) return false;
	if (vMaxs.z >= bigBox.vMaxs.z) return false;
	return true;
}


M_INLINE
bool sBBox::Intersect(const sBBox & bb1, const sBBox & bb2)
{
	if (bb1.vMins.x > bb2.vMaxs.x || bb1.vMaxs.x < bb2.vMins.x ||
		bb1.vMins.y > bb2.vMaxs.y || bb1.vMaxs.y < bb2.vMins.y ||
		bb1.vMins.z > bb2.vMaxs.z || bb1.vMaxs.z < bb2.vMins.z)
	{
		ClearBounds();
		return false;
	}
	vMins.x = std::max(bb1.vMins.x, bb2.vMins.x);
	vMins.y = std::max(bb1.vMins.y, bb2.vMins.y);
	vMins.z = std::max(bb1.vMins.z, bb2.vMins.z);
	vMaxs.x = std::min(bb1.vMaxs.x, bb2.vMaxs.x);
	vMaxs.y = std::min(bb1.vMaxs.y, bb2.vMaxs.y);
	vMaxs.z = std::min(bb1.vMaxs.z, bb2.vMaxs.z);
	return true;
}

M_INLINE
bool sBBox::Union(const sBBox & bb1, const sBBox & bb2)
{
	vMins.x = std::min(bb1.vMins.x, bb2.vMins.x);
	vMins.y = std::min(bb1.vMins.y, bb2.vMins.y);
	vMins.z = std::min(bb1.vMins.z, bb2.vMins.z);
	vMaxs.x = std::max(bb1.vMaxs.x, bb2.vMaxs.x);
	vMaxs.y = std::max(bb1.vMaxs.y, bb2.vMaxs.y);
	vMaxs.z = std::max(bb1.vMaxs.z, bb2.vMaxs.z);
	return !IsEmpty();
}

M_INLINE
	bool sBBox::operator == (const sBBox &ref) const { return vMins == ref.vMins && vMaxs == ref.vMaxs; }

M_INLINE
	bool sBBox::operator != (const sBBox &ref) const { return vMins != ref.vMins || vMaxs != ref.vMaxs; }

M_INLINE
void sBBox::ExpandIfNear(const sBBox &ref)
{
	const float fEps = 0.0000002f;

	if (ref.vMaxs.x > vMaxs.x && ref.vMaxs.x - vMaxs.x <= fEps) vMaxs.x = ref.vMaxs.x;
	if (vMins.x > ref.vMins.x && vMins.x - ref.vMins.x <= fEps) vMins.x = ref.vMins.x;

	if (ref.vMaxs.y > vMaxs.y && ref.vMaxs.y - vMaxs.y <= fEps) vMaxs.y = ref.vMaxs.y;
	if (vMins.y > ref.vMins.y && vMins.y - ref.vMins.y <= fEps) vMins.y = ref.vMins.y;

	if (ref.vMaxs.z > vMaxs.z && ref.vMaxs.z - vMaxs.z <= fEps) vMaxs.z = ref.vMaxs.z;
	if (vMins.z > ref.vMins.z && vMins.z - ref.vMins.z <= fEps) vMins.z = ref.vMins.z;
}


// ------------------------------------------------------------------------ //
// Rectangle
// ------------------------------------------------------------------------ //

M_INLINE
float sFRect::Width() const {return right - left;}

M_INLINE
float sFRect::Height() const {return bottom - top;}

M_INLINE
Vec2 sFRect::Size() const {return IsEmpty() ? Vec2::Null : Vec2(right - left, bottom - top);}

M_INLINE
Vec2& sFRect::Mins() {return *(Vec2 *)&left;}

M_INLINE
Vec2& sFRect::Maxs() {return *(Vec2 *)&right;}

M_INLINE
const Vec2& sFRect::Mins() const {return *(Vec2 *)&left;}

M_INLINE
const Vec2& sFRect::Maxs() const {return *(Vec2 *)&right;}

M_INLINE
Vec2 sFRect::Center() const {return Vec2((left + right) * 0.5f, (top + bottom) * 0.5f);}

M_INLINE
bool sFRect::IsEmpty() const {return (left >= right) || (top >= bottom);}

M_INLINE
bool sFRect::Test(const Vec2 & pt) const
{
	return (pt.x >= left) && (pt.x <= right) && (pt.y >= top) && (pt.y <= bottom);
}

M_INLINE
void sFRect::Set(float x1, float y1, float x2, float y2)
{
	left = x1;	top = y1;
	right = x2;	bottom = y2;
}

M_INLINE
bool sFRect::operator == (const sFRect & rc) const
{
	return (left == rc.left) && (right == rc.right) && (top == rc.top) && (bottom == rc.bottom);
}

M_INLINE
bool sFRect::operator != (const sFRect & rc) const
{
	return (left != rc.left) || (right != rc.right) || (top != rc.top) || (bottom != rc.bottom);
}

M_INLINE
void sFRect::Inflate(float x, float y)
{
	left -= x;	right += x;
	top -= y;	bottom += y;
}

M_INLINE
void sFRect::Inflate(const Vec2 & size)
{
	left -= size.x;	right += size.x;
	top -= size.y;	bottom += size.y;
}

M_INLINE
void sFRect::Deflate(float x, float y)
{
	left += x;	right -= x;
	top += y;	bottom -= y;
}

M_INLINE
void sFRect::Deflate(const Vec2 & size)
{
	left += size.x;	right -= size.x;
	top += size.y;	bottom -= size.y;
}

M_INLINE
void sFRect::Normalize()
{
	if (left > right) std::swap(left, right);
	if (top > bottom) std::swap(top, bottom);
}

M_INLINE
void sFRect::Offset(const Vec2 & pt)
{
	left += pt.x;	right += pt.x;
	top += pt.y;	bottom += pt.y;
}

M_INLINE
void sFRect::MoveToY(float y)
{
	bottom = Height() + y;
	top = y;
}

M_INLINE
void sFRect::MoveToX(float x)
{
	right = Width() + x;
	left = x;
}

M_INLINE
void sFRect::MoveToXY(const Vec2 & pt)
{
	MoveToX(pt.x);
	MoveToY(pt.y);
}

M_INLINE
bool sFRect::Intersect(const sFRect & rc1, const sFRect & rc2)
{
	if (rc1.left >= rc2.right || rc1.right <= rc2.left ||
		rc1.top >= rc2.bottom || rc1.bottom <= rc2.top)
	{
		Set(0, 0, 0, 0);
		return false;
	}
	left = std::max(rc1.left, rc2.left);
	top = std::max(rc1.top, rc2.top);
	right = std::min(rc1.right, rc2.right);
	bottom = std::min(rc1.bottom, rc2.bottom);
	return true;
}

M_INLINE
bool sFRect::Union(const sFRect & rc1, const sFRect & rc2)
{
	left = std::min(rc1.left, rc2.left);
	top = std::min(rc1.top, rc2.top);
	right = std::max(rc1.right, rc2.right);
	bottom = std::max(rc1.bottom, rc2.bottom);
	return !IsEmpty();
}

M_INLINE
void sFRect::ClearBounds()
{
	left = top = FLT_MAX;
	right = bottom = -FLT_MAX;
}

M_INLINE
void sFRect::AddToBounds(const Vec2 &v)
{
	if (left > v.x)		left = v.x;
	if (right < v.x)	right = v.x;
	if (top > v.y)		top = v.y;
	if (bottom < v.y)	bottom = v.y;
}
