//
//  CRGraphicsObjectBase.h
//  CRGraphics
//
//  Created by Yaroslav Glushchenko on 10/14/10.
//  Copyright 2010 Corner-A. All rights reserved.
//

#import "CRGraphicsDraggableView.h"

@class CRGraphicsObjectBase;
@class CRGraphicsObjectRotator;

extern NSString* CRGraphicsObjectAnimationTickNotification;

//////////////////////////////////////////////////////////////////////
// Graphics object delegate
//////////////////////////////////////////////////////////////////////
@protocol CRGraphicsObjectDelegate

- (NSValue *)constraintValue:(NSValue *)value ofObject:(id)object named:(NSString *)named;

@end

//////////////////////////////////////////////////////////////////////
// Delayed update info enum
//////////////////////////////////////////////////////////////////////
typedef enum {
	CRGeometryUpdateMove,
	CRGeometryUpdateSize,
	CRGeometryUpdateRotation
} CRGraphicsObjectGeometryUpdateType;

//////////////////////////////////////////////////////////////////////
// Base graphics object
//////////////////////////////////////////////////////////////////////
@interface CRGraphicsObjectBase : CRGraphicsDraggableView<CRGraphicsObjectDelegate>
{
@private
	CRTwoWayBindingManager*	bindingManager;
	
@protected
	BOOL					ignoreConstraint;
	BOOL					constraintInSuperview;
	BOOL					transformGraphicsRect;
	
	NSRect					filterSpaceRect;
	BOOL					setRectSilent;			// setGraphicsRect posts no KVC notifications with this flag
	BOOL					preventRectsChain;		// helps to void filterRect -> graphicsRect -> filterRect change chains
	BOOL					ignoreHandler;
	
	NSRect					workingArea;			// working area on the view
	CGRect					safeArea;				// safe area: the working zone plus safe border
	CGRect					processingArea;			// photo area
	NSMutableArray*			graphicsObjects;		// list of graphics objects
	
	int						dashPhase;
	
	id						temporaryContent;		// content to draw inside the item
}

@property (nonatomic, readonly) BOOL							wantsAnimation;
@property (nonatomic, assign)	BOOL							constraintInSuperview;
@property (nonatomic, assign)	BOOL							transformGraphicsRect;
@property (nonatomic, assign)	NSRect							filterSpaceRect;

@property (nonatomic, assign)	NSRect							workingArea;
@property (nonatomic, assign)	CGRect							safeArea;
@property (nonatomic, assign)	CGRect							processingArea;

@property (nonatomic, assign)	id<CRGraphicsObjectDelegate>	delegate;

// converters
- (NSSize)scaleToView;
- (NSSize)scaleFromView;
- (NSSize)scaleToWorking;
- (NSSize)scaleFromWorking;
- (NSRect)scaleRect:(NSRect)rect withScale:(NSSize)scale;

- (NSRect)filterSpaceToViewSpace:(NSRect)rect;
- (NSRect)viewSpaceToFilterSpace:(NSRect)rect;
- (NSRect)viewSpaceToWorkingSpace:(NSRect)rect;
- (NSRect)workingSpaceToViewSpace:(NSRect)rect;
- (NSRect)filterSpaceToWorkingSpace:(NSRect)rect;
- (NSRect)workingSpaceToFilterSpace:(NSRect)rect;

- (NSRect)processingRect;
- (NSRect)safeRect;

- (BOOL)antialiasing;
- (NSImageInterpolation)imageInterpolation;

// sub-items
- (void)addGraphicsObject:(CRGraphicsObjectBase *)anObject;
- (void)removeGraphicsObject:(CRGraphicsObjectBase *)anObject;
- (void)removeAllGraphicsObjects;
- (void)updateGraphicsObjectSpace;

// delayed geometry updates
- (void)controlPointDraggingStarted;
- (void)controlPointDraggingFinished;
- (void)draggingStarted;
- (void)draggingFinished;

- (BOOL)groupedRectsUpdate;
- (void)prepareForDelayedDrag;
- (void)finalizeDelayedDrag;

- (void)prepareTemporaryContent:(CRGraphicsObjectGeometryUpdateType)updateType;
- (void)finalizeTemporaryContent;
- (void)drawTemporaryContent;

// animation
- (BOOL)wantsAnimation;
- (NSTimeInterval)animationInterval;
- (void)invalidateAnimation;
- (void)timerTick:(id)notification;

// auxilliary
- (NSRect)boundingRect;			// bounding rect for the frame rect with the rotation applied
- (NSRect)innerRect;			// bounds fixed with an inset
- (NSRect)innerBoundingRect;	// bounding rect for rotated inner rect
- (NSRect)boundingRectForRect:(NSRect)rect angle:(CGFloat)angle;

- (NSPoint)constraintOrigin:(NSPoint)origin;
- (NSSize)constraintSize:(NSSize)size;
- (NSRect)constraintRect:(NSRect)rect;

- (void)updateGraphicsRect;
- (void)updateFilterSpaceRect;

- (NSRect)convertRectToWorkingSpace:(NSRect)rect;
- (NSRect)convertRectToFilterSpace:(NSRect)rect;

- (int)dashSize;
- (int)dashPhase;

// frame replacement, deals with center rotation automatically
- (NSRect)graphicsRect;
- (NSRect)graphicsRectWithInset;
- (void)setGraphicsRect:(NSRect)rect;

- (void)spaceChanged;

@end